{+doc
+NOMBRE: memorias
+CREACION: 1.7.94
+AUTORES: rch
+REGISTRO:
+TIPO: Unidad Pascal.
+PROPOSITO:  implementacin de una memoria digital.
+PROYECTO: SiMEEP

+REVISION:
+AUTOR: rch
+DESCRIPCION:

	Definicin del objeto (TMemoria). TMemoria esta pensada para funcionar
como memoria para un canal de muestreo digital de una seal.

**********************
	DEFINICION
**********************

TMemoria= object( TVectR )



**********************
constructor init(
**********************

PARAMETROS DE CONSTRUCCION
--------------------------

PeriodoDeMuestreo: NReal
	Tiempo entre muestras consecutivas.

NDatos: integer
	Tamao del buffer en nmero de muestras. Dureante la inicializacin se
	reserva memoria para almacenar esa cantidad de muestras.

t1, x1: NReal
	Instante y valor de la primera muestra.


**************************
destructor done; virtual;
**************************

	Devuelve al sistema la memoria reservada para almacenar las muestras.

**************************************
procedure Tic( t, x: NReal ); virtual;
**************************************
	Realiza el muestreo uniforme de la seal x(t). Al llamar este
	procedimiento, se traza una lnea recta entre el ltimo punto ingresado
	y el punto x(t) y si corresponde se realiza el muestreo y se almacenan los
	valores.

***********************
procedure FFT; virtual;  (Fast Fourier Transform)
***********************
	Realiza una FFT sobre las muestras de la memoria. Si la memoria
	ya haba sido inicializada para manejos en el dominio de la frecuencia
	(ver SF_init) se actualiza la serie de Fourier (discreta) de la seal
	almacenada en la memoria. De esta manera, se mantiene una doble
	representacin de la seal en el dominio de la frecuencia y en el
	dominio del tiempo.

**************************
procedure SF_init; virtual;
**************************
	Reserva el espacio adicional requerido para almacenar la representacion
	de la seal en el dominio de la frecuencia. La representacin se
	realiza por el trmino constante (a0) el vector de coeficientes de
	la serie de cosenos (an) y el vector de coeficientes de la serie de
	senos (bn).

***************************
procedure SF_done; virtual;
***************************
	Devuelve la memoria adicional reservada mediante SF_init.

***********************************************
function PotenciaArmonica( an: integer): NReal;
***********************************************
	Esta funcin retorna la potencia del armnico de orden (an) en la seal.
	Para que el resultado tenga sentido, primero es necesario haber realizado
	una FFT para que la representacin en el dominio de la frecuencia est
	actualizada.

********************************************
procedure RetrocesoBarrido( deltaT: NReal );
********************************************
	Retrocede el tiempo de barrido de la memoria en deltaT. Este mtodo est
	pensado para permitir durante la simulacin de mltiples perodos de un
	sistema retroceder el tiempo de simulacin al final de cada pantalla
	grfica y comenzar un nuevo barrido. Esto permite utilizar los objetos
	TMemoria, junto con la unidad Traxp para simular el comportamiento de
	un osciloscopio digital y ver en pantalla, una seal que se actualiza
	barrido a barrido.


	private
		SF_Inicializada: boolean;
		t0: NReal;
		x0: NReal;
		dt: NReal;
		ts: NReal;
		is: integer;
		ac, bc: LVR1Ptr;
		a0:NReal;
		LongBarrido: NReal;
	end;


-doc}

unit memorias;

interface
uses
	classes, xMatDefs, MatReal, wRFFTI01, wRFFTF01, horrores;
type
	LVR1Ptr= ^LVR1;
	LVR1 = array[1..6000] of NReal;

type

	TMemoria= class ( TVectR )
		constructor Create_init(
			PeriodoDeMuestreo: NReal;
			NDatos: integer;
			t1, x1: NReal );

	        procedure Free; virtual;

		constructor Create_Load( var s: TStream );
		procedure Store( var s: TStream ); virtual;
		{
		function Promedio: NReal;
		 }
		procedure Tic( t, x: NReal ); virtual;
		procedure FFT; virtual;
		procedure SF_init; virtual;
		procedure SF_store( var s: TStream ); virtual;
		procedure SF_Load( var s: TStream ); virtual;
		procedure SF_done; virtual;
		function ValorMedio: NReal;
		function PotenciaArmonica( an: integer): NReal;
		procedure RetrocesoBarrido( deltaT: NReal );
		function DistorsionArmonica_25: NReal;

		procedure FijarVentanaDeTiempo( xt0, xt1: NReal); virtual;


	private
		SF_Inicializada: boolean;
		t0: NReal;
		x0: NReal;
		dt: NReal;
		ts: NReal;
		is_: integer;
		ac, bc: LVR1Ptr;
		a0:NReal;
		LongBarrido: NReal;
	end;



implementation


constructor TMemoria.Create_init(
			PeriodoDeMuestreo: NReal;
			NDatos: integer;
			t1, x1: NReal );

begin

	inherited Create_Init(NDatos);
	Ceros;
	t0:= t1;
	x0:= x1;
	dt:= PeriodoDeMuestreo;
	ts:=t0+dt;
	is_:= 1;
	LongBarrido:= NDatos* dt;
	pon_e(is_, x1);
	if is_ = N then	is_:= 1
	else is_:= is_+1;
	SF_Inicializada:= false;
end;


procedure TMemoria.FijarVentanaDeTiempo( xt0, xt1: NReal);
begin
     t0:=xt1;
     LongBarrido:= xt1-xt0;
		 dt:= LongBarrido/N;
end;



procedure TMemoria.RetrocesoBarrido( DeltaT: NReal );
begin
	ts:= ts-DeltaT;
	t0:= t0-DeltaT;
end;

procedure TMemoria.Tic( t, x: NReal );
var
	xs: NReal;
begin
	if t>= ts then
	while t>= ts do
	begin
		if EsCero(t - t0) then xs:=x0
		else
			xs:=x0+(x-x0)/(t-t0)*(ts-t0);
		pon_e(is_, xs);
		x0:=xs;t0:=ts;
		ts:=ts+dT;
		if is_= N then	is_:= 1
		else is_:= is_+1;
	end
	else
	begin
		x0:=x;t0:=t;
	end
end;

procedure TMemoria.SF_Init;
var
	TamVectsCoefs: word;

begin
	TamVectsCoefs:= (N div 2)* SizeOf(NReal);
	GetMem(ac, TamVectsCoefs);
	GetMem(bc, TamVectsCoefs);
	SF_Inicializada:= true;
end;

procedure TMemoria.SF_Store( var s: TStream );
var
	TamVectsCoefs: word;

begin
	TamVectsCoefs:= (N div 2)* SizeOf(NReal);
	s.write( a0, SizeOf( a0 ));
	s.write( ac^, TamVectsCoefs );
	s.write( bc^, TamVectsCoefs );
end;


procedure TMemoria.Sf_Load( var s: TStream );
var
	TamVectsCoefs: word;

begin
	TamVectsCoefs:= (N div 2)* SizeOf(NReal);
	s.read( a0, SizeOf( a0 ));
	s.read( ac^, TamVectsCoefs );
	s.read( bc^, TamVectsCoefs );
end;


procedure TMemoria.SF_done;
var
	TamVectsCoefs: integer;
begin
	TamVectsCoefs:= (N div 2)* SizeOf(NReal);
	FreeMem(bc, TamVectsCoefs);
	FreeMem(ac, TamVectsCoefs);
	SF_Inicializada:= false;
end;

procedure TMemoria.FFT;
var
	datos:LVR1Ptr; { NData }

begin
	if not SF_Inicializada then SF_Init;
	datos:= @pv[1];
	wRFFTI01.Init(N);
	FFTF(datos^,a0,ac^,bc^);
end;

function TMemoria.PotenciaArmonica( an: integer): NReal;
begin
	if (an> N div 2)or(an<0) then error(' indice de armnica fuera de rango');
	if an = 0 then PotenciaArmonica:= a0*a0
	else
		PotenciaArmonica:= ((ac^[an]*ac^[an])+(bc^[an]*bc^[an]))/2;
end;

function TMemoria.ValorMedio: NReal;
begin
	ValorMedio:= a0;
end;

function TMemoria.DistorsionArmonica_25: NReal;
var
	k, kmax: integer;
	acum: NReal;
begin
	kmax:= n div 2;
	if kmax > 25 then kmax := 25;
	acum:= 0;
	for k := 2 to kmax do
		acum:= acum + PotenciaArmonica(k);
	acum:= sqrt(acum/(acum+ PotenciaArmonica(1)));
	DistorsionArmonica_25:= acum;
end;

procedure TMemoria.Free;
begin
  if SF_Inicializada then SF_done;
  inherited Free;
end;


constructor TMemoria.Create_Load( var s: TStream );
begin
	inherited Create_Load(s);
	s.read(t0, sizeOf(t0));
	s.read(x0, sizeOf(x0));
	s.read(	dt, sizeOf(dt));
	s.read(	ts, sizeOf(ts) );
	s.read(	is_, sizeOf(is_) );
	s.read( LongBarrido, SizeOf(LongBarrido));
	s.read( SF_Inicializada, SizeOf( SF_Inicializada ));
	if SF_Inicializada then
	begin
		SF_init;
		SF_Load(s);
	end;
end;

procedure TMemoria.Store( var s: TStream );
begin
	inherited Store(s);
	s.write(t0, sizeOf(t0));
	s.write(x0, sizeOf(x0));
	s.write(	dt, sizeOf(dt));
	s.write(	ts, sizeOf(ts) );
	s.write(	is_, sizeOf(is_) );
	s.write( LongBarrido, SizeOf(LongBarrido));
	s.write( SF_Inicializada, SizeOf( SF_Inicializada ));
	if SF_Inicializada then
		SF_Store(s);
end;


end.
