unit uSuministroCombustible_TakeOrPay;

interface

uses
  Math, xMatDefs, uSimplex, uGlobs, uFechas, uEstados, Classes,
  uActores, uFichasLPD, uNodos,
  uGeneradores,
  uSuministroCombustible,
  uCosa,
  uFuentesAleatorias,
  uconstantesSimSEE, SysUtils,
  uConversions,
  uAgenda;

type

  { TFichaSuministroCombustible_TakeOrPay }

  TFichaSuministroCombustible_TakeOrPay = class(TFichaSuministroCombustible)
  public

    (**************************************************************************)
    (*               A T R I B U T O S   P E R S I S T E N T E S              *)
    (**************************************************************************)

    T_ToP: NInt; // [dias] Tiempo de Take or Pay.
    V_ToP: NReal; // [unidad e Volumen] Volumen comprometido por el contrato Take Or Pay
    P_ToP: NReal;
    // [U$] Precio por Unidad de Volumen de combustible consumido por encima del volumen comprometido por Take or Pay
    P_ExtraToP: NReal;
    //En el caso de que los embarques no sean periódicos,
    //indica los días desde el comienzo de la ficha en que ingresan embarques.
    DiasIngresoEmbarques: TDAOfNInt;
    strDiasIngresoEmbarques: string;
    C_Max: NReal; // [unidad e Volumen/h] Caudal máximo del suministro
    limites_postizados: boolean;
    manejoBOG: boolean;
    // K_BOG: Porcentaje de volumen del tanque que se evapora en BOG
    K_BOG: NReal;
    // K_GSS: Volumen de GN en m3 requerido por la planta para operar, incluso si no esta emitiendo gas
    K_GSS: NReal;
    // K_RG: Porcentaje de GN que se debe utilizar para regasificar una unidad de GNL
    K_RG: NReal;
    // K_RC: Porcentaje de BOG que se puede recondensar cuando la planta emite gas una unidad de GN
    K_RC: NReal;
    // K_RL: Ratio de BOG que se puede relicuar sobre BOG que se utiliza en el proceso de relicuado (relicuado + utilizado para dar energía al proceso)
    K_RL: NReal;
    Relicuado: boolean; // indica si se tiene una planta de relicuado
    agPerturbacionDias: TFuenteAleatoria;
    agBornePerturbacionDias: string;

    (**************************************************************************)

    V_Efectivo: NReal;
    // [U$] Precio por Unidad de Volumen de combustible dentro del comprometido por Take or Pay
    EmbarquesPeriodicos: boolean; //Indica si los embarques son periódicos o no.
    agenda: TAgendaEmbarque;
    agNroBornePerturbacionDias: integer;

    constructor Create(capa: integer; fecha: TFecha; periodicidad: TPeriodicidad;
      nombrecombustible: string; T_ToP: NInt; V_ToP: NReal; P_ToP: NReal;
      P_ExtraToP: NReal; C_Max: NReal; limitePorPoste: boolean;
      EmbarquesPeriodicos: boolean; strDiasIngresoEmbarques: string;
      indicePreciosPorCombustible: TFuenteAleatoria; bornePreciosPorCombustible: string;
      manejoBOG: boolean; K_BOG: NReal; K_GSS: NReal; K_RG: NReal; K_RC: NReal;
      K_RL: NReal; Relicuado: boolean; agPerturbacionDias: TFuenteAleatoria;
      agBornePerturbacionDias: string);
     
    constructor Create_ReadFromText(f: TArchiTexto); override;
    procedure WriteToText(f: TArchiTexto); override;
    

    function infoAd_: string; override;
    function GetStrDiasIngresoEmbarques(): string;
    procedure GetFechaSiguienteToP(fechaActual: TDateTime;
      var fecha_ToP_sig: TDateTime; globs: TGlobs);
    procedure Free; override;

    



    procedure AfterInstantiation; override;
  published

  



































  end;

type

  { TSuministroCombustible_TakeOrPay }

  TSuministroCombustible_TakeOrPay = class(TSuministroCombustible)

  (*
  Esta clase representará posibles suministros de combustible para
  los generadores termicos
  *)
  public

    pa: TFichaSuministroCombustible_TakeOrPay;
    //Variables de Estado

    VDisp_ToP_Ini: NReal;  //Volumen disponible de ToP (varia entre V_ToP y 0
    NDisc: NInt; //Cantidad de discretizaciones del volumen

    // El valor de la variable de estado X_V en el paso k y representa
    // el volumen restante (sin consumir) del contrato de ToP
    // vigente al inicio del paso  k.
    //Es lo que Tengo disponible en el tanque al inicio del paso
    //  (Xs_VDisp_ToP  es el estado al final del paso)
    X_VDisp_ToP: NReal;
    Xs_VDisp_ToP: NReal;

    BOG_T, BOG_GSS, BOG_RG, BOG_RC, BOG_RL, BOG_GRL, BOG_Vent, BSO: NReal;


    // derivada del costo futuro respecto del Volumen Disponible ToP
    dCFdV: NReal;

    cvExtraToP: NReal;  //Costo Variable de lo consumido por sobre del ToP
    cvToP: NReal;  //Costo Variable del ToP (dCF/dV)
    cfToP: NReal;  //Costo Fijo del ToP (VToP * PTop)

    EmbarquesPeriodicos: boolean;

    //Variables calculadas en el paso de tiempo durante la simulación
    //Volumen que despacho de combustible por debajo del disponible
    //en el paso de simulación (por lo tanto no aporta un costo extra al paso)
    VConsumido_ToP: NReal;

    //Volumen que despacho de combustible por sobre el disponible.
    //En este caso aporto al paso un costo P_ExtraToP por cada unidad de combustible
    VConsumido_Extra_Top: NReal;

    //Este es el volumen de combustible del ToP que se "tiró" al no ser consumido.
    VSobrante_ToP: NReal;


    V_ToP_sig: NReal;  //Volumen ToP del proximo embarque;

    fase: integer; // 1 = Inicio del TOP; 0= Dentro del periodo de TOP

    costoToP, costoExtraToP: NReal;
    indice_res_ETop, indice_var_ETop: integer;

    constructor Create(capa: integer; nombre: string; nombreCombustible: string;
      nacimiento, muerte: TFecha; lpdUnidades, lpd: TFichasLPD;
      VDisp_ToP_Ini: NReal; NDisc: NInt); virtual;


     
    constructor Create_ReadFromText(f: TArchiTexto); override;
    procedure WriteToText(f: TArchiTexto); override;
    

    function get_ires_volumen: NInt; override;
    function get_precio_combustible: NReal; override;
    function get_combustible: TCombustibleSGE; override;
    function restricciones_postizadas: boolean; override;

    class function TipoFichaLPD: TClaseDeFichaLPD; virtual; abstract;
    //      procedure sim_PrintResultados_Encab_PotFirme(var fsal: textfile; kencab: integer ); virtual;
    //      procedure sim_PrintResultados_PotFirme(var fsal: textfile); virtual;
    procedure Free; override;
    class function DescClase: string; override;


    procedure PrepararMemoria(globs: TGlobs); override;
    procedure RegistrarParametrosDinamicos; override;


    procedure SorteosDelPaso(sortear: boolean); override;
    procedure PrepararPaso_ps; override;

    procedure Sim_Cronica_Inicio; override;

    procedure opt_nvers(var ivar, ivae, ires: integer); override;

    procedure opt_cargue(s: TSimplex); override;

    procedure EvolucionarEstado; override;

    procedure opt_fijarRestriccionesDeCaja(s: TSimplex); override;

    procedure opt_leerSolucion(s: TSimplex); override;

    function getNombreVar(ivar: integer; var nombre: string): boolean; override;
    function getNombreRes(ires: integer; var nombre: string): boolean; override;

    procedure PosicionarseEnEstrellita; override;
    procedure optx_nvxs(var ixr, ixd, iauxr, iauxd: integer); override;

    // imprime tabla de delta costo futuro de la variable de estado..
    procedure opt_PrintResultados_Encab(var fsal: textfile); override;
    procedure opt_PrintResultados(var fsal: textfile); override;

    procedure optx_RegistrarVariablesDeEstado(adminEstados: TAdminEstados); override;
    procedure ActualizarEstadoGlobal(flg_Xs: boolean); override;

    //      procedure sim_PrintResultados_Encab(var fsal: textfile; kencab: integer ); override;
    //      procedure sim_PrintResultados(var fsal: textfile); override;

    procedure PubliVars; override;
    //          function varsPSimRes3PorDefecto: TStringList; override;

    



    procedure AfterInstantiation; override;

  published
    




  end;

procedure AlInicio;
procedure AlFinal;
procedure TSuministroCombustible_TakeOrPay_cambioFichaPD(Actor: TCosa);

implementation


//------------------------------
// Métodos de TFichaSuministroCombustible_TakeOrPay
//==============================
constructor TFichaSuministroCombustible_TakeOrPay.Create(capa: integer;
  fecha: TFecha; periodicidad: TPeriodicidad; nombrecombustible: string;
  T_ToP: NInt; V_ToP: NReal; P_ToP: NReal; P_ExtraToP: NReal; C_Max: NReal;
  limitePorPoste: boolean; EmbarquesPeriodicos: boolean;
  strDiasIngresoEmbarques: string; indicePreciosPorCombustible: TFuenteAleatoria;
  bornePreciosPorCombustible: string; manejoBOG: boolean; K_BOG: NReal;
  K_GSS: NReal; K_RG: NReal; K_RC: NReal; K_RL: NReal; Relicuado: boolean;
  agPerturbacionDias: TFuenteAleatoria; agBornePerturbacionDias: string);

begin
  inherited Create(capa, fecha, periodicidad, nombrecombustible,
    indicePreciosPorCombustible, bornePreciosPorCombustible);

  self.T_ToP := T_ToP;
  self.V_ToP := V_ToP;
  self.P_ToP := P_ToP;
  self.P_ExtraToP := P_ExtraToP;
  self.C_Max := C_Max;
  self.limites_postizados := limitePorPoste;
  self.EmbarquesPeriodicos := EmbarquesPeriodicos;
  //  self.DiasIngresoEmbarques := StrToDAOfNInt_(strDiasIngresoEmbarques, ';');
  self.manejoBOG := manejoBOG;
  self.K_BOG := K_BOG;
  self.K_GSS := K_GSS;
  self.K_RG := K_RG;
  self.K_RC := K_RC;
  self.K_RL := K_RL;
  self.Relicuado := Relicuado;

  self.agenda := TAgendaEmbarque.Create(capa, fecha, self.T_ToP,
    self.V_ToP, self.P_ToP, self.EmbarquesPeriodicos, strDiasIngresoEmbarques);

  self.agPerturbacionDias := agPerturbacionDias;
  self.agBornePerturbacionDias := agBornePerturbacionDias;

end;

 
constructor TFichaSuministroCombustible_TakeOrPay.Create_ReadFromText(f: TArchiTexto);
begin
  if f.version < 44 then
  begin
    inherited Create_ReadFromText(f);
    f.IniciarLecturaRetrasada;
    f.rd('T_ToP', T_ToP);
    f.rd('V_ToP', V_ToP);
    f.rd('P_ToP', P_ToP);
    f.rd('P_ExtraToP', P_ExtraToP);
    f.EjecutarLectura;

    EmbarquesPeriodicos := True;
    setlength(DiasIngresoEmbarques, 0);
    self.C_Max := -1;
    self.limites_postizados := False;
    self.manejoBOG := False;
    self.K_BOG := 0;
    self.K_GSS := 0;
    self.K_RG := 0;
    self.K_RC := 0;
    self.K_RL := 0;
    self.Relicuado := False;
    self.agPerturbacionDias := nil;
    self.agBornePerturbacionDias := '';

  end
  else if f.version < 45 then
  begin
    inherited Create_ReadFromText(f);
    f.IniciarLecturaRetrasada;
    f.rd('T_ToP', T_ToP);
    f.rd('V_ToP', V_ToP);
    f.rd('P_ToP', P_ToP);
    f.rd('P_ExtraToP', P_ExtraToP);
    f.rd('EmbarquesPeriodicos', EmbarquesPeriodicos);
    f.rd('DiasIngresoEmbarques', strDiasIngresoEmbarques);
    f.EjecutarLectura;

    //    self.DiasIngresoEmbarques := StrToDAOfNInt_(strDiasIngresoEmbarques, ';');
    self.agenda := TAgendaEmbarque.Create(capa, fecha, self.T_ToP,
      self.V_ToP, self.P_ToP, self.EmbarquesPeriodicos, strDiasIngresoEmbarques);
    self.C_Max := -1;
    self.limites_postizados := False;
    self.manejoBOG := False;
    self.K_BOG := 0;
    self.K_GSS := 0;
    self.K_RG := 0;
    self.K_RC := 0;
    self.K_RL := 0;
    self.Relicuado := False;
    self.agPerturbacionDias := nil;
    self.agBornePerturbacionDias := '';
  end
  else
  begin
    inherited Create_ReadFromText(f);
    f.IniciarLecturaRetrasada;
    f.rd('T_ToP', T_ToP);
    f.rd('V_ToP', V_ToP);
    f.rd('P_ToP', P_ToP);
    f.rd('P_ExtraToP', P_ExtraToP);
    f.rd('EmbarquesPeriodicos', EmbarquesPeriodicos);
    f.rd('DiasIngresoEmbarques', strDiasIngresoEmbarques);
    f.rd('C_Max', C_Max);
    f.rd('Limites_Postizados', limites_postizados);
    f.rd('Manejo_BOG', manejoBOG);
    f.rd('BOG_hora', K_BOG);
    f.rd('Servicios_generales', K_GSS);
    f.rd('RG_por_emision', K_RG);
    f.rd('BOG_Recondensado', K_RC);
    f.rd('Ratio_RL', K_RL);
    f.rd('Hay_Relicuado', Relicuado);
    f.rdReferencia('agPerturbacionDias', TCosa(agPerturbacionDias), self);
    f.rd('agBornePerturbacionDias', agBornePerturbacionDias);
    f.EjecutarLectura;

    //    self.DiasIngresoEmbarques := StrToDAOfNInt_(strDiasIngresoEmbarques, ';');
    self.agenda := TAgendaEmbarque.Create(capa, fecha, self.T_ToP,
      self.V_ToP, self.P_ToP, self.EmbarquesPeriodicos, strDiasIngresoEmbarques);
  end;
end;

procedure TFichaSuministroCombustible_TakeOrPay.WriteToText(f: TArchiTexto);
begin
  inherited WriteToText(f);
  f.wr('T_ToP', T_ToP);
  f.wr('V_ToP', V_ToP, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES);
  f.wr('P_ToP', P_ToP, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES);
  f.wr('P_ExtraToP', P_ExtraToP, uconstantesSimSEE.CF_PRECISION,
    uconstantesSimSEE.CF_DECIMALES);
  f.wr('EmbarquesPeriodicos', EmbarquesPeriodicos);
  //  f.wr('DiasIngresoEmbarques', GetStrDiasIngresoEmbarques());
  //  agenda.WriteToText(f);
  f.wr('DiasIngresoEmbarques', agenda.ToString());
  f.wr('C_Max', C_Max);
  f.wr('Limites_Postizados', limites_postizados);
  f.wr('Manejo_BOG', manejoBOG);
  f.wr('BOG_hora', K_BOG);
  f.wr('Servicios_generales', K_GSS);
  f.wr('RG_por_emision', K_RG);
  f.wr('BOG_Recondensado', K_RC);
  f.wr('Ratio_RL', K_RL);
  f.wr('Hay_Relicuado', Relicuado);
  f.wrReferencia('agPerturbacionDias', agPerturbacionDias);
  f.wr('agBornePerturbacionDias', agBornePerturbacionDias);
end;




function TFichaSuministroCombustible_TakeOrPay.GetStrDiasIngresoEmbarques(): string;
begin
  Result := agenda.ToString();
  //  Result := DAOfNIntToStr_(self.DiasIngresoEmbarques, ';');
end;

//function TFichaSuministroCombustible_TakeOrPay.GetFechaSiguienteToP(fechaActual : TDateTime; globs: TGlobs) : TDateTime;
procedure TFichaSuministroCombustible_TakeOrPay.GetFechaSiguienteToP(
  fechaActual: TDateTime; var fecha_ToP_sig: TDateTime; globs: TGlobs);
var
  cntDias, diasAux, i, diasSiguienteEmbarque: integer;
  fecha_ToP_ant: TDateTime;
  //  fecha_ToP_sig : TDateTime;
  s: string;

begin
  if (T_ToP = -1) then
  begin
    //    exit(fechaActual + 99999999); //no va a recibir cargamentos o sea el tanque no juega
    P_ToP := 0;
    V_ToP := 0;
    fecha_ToP_sig := fechaActual + 99999999;
  end
  else
  begin
    if self.EmbarquesPeriodicos then
    begin
      //Calculamos cuando fué el último embarque (más reciente) para la fecha de inicio del paso.
      //Esto puede dar la fecha de vigencia de la ficha (cuando comienza a regir una ficha entra combustible)
      //o una fecha posterior tal que es múltiplo de T_ToP
      cntDias := Trunc(fechaActual) - Trunc(self.Fecha.dt);
      cntDias := cntDias mod self.T_ToP;
      fecha_ToP_ant := fechaActual - cntDias;

      //Calculamos la siguiente fecha de ingreso de TakeOrPay y la hacemos igual a la fecha anterior + T dias
      if fecha_ToP_ant = Trunc(fechaActual) then
        fecha_ToP_sig := Trunc(fechaActual)
      else
        fecha_ToP_sig := fecha_ToP_ant + self.T_ToP;
      //        P_ToP := self.P_ToP;
      //        V_ToP := self.V_ToP;

    end
    else
    begin
      agenda.GetFechaSiguienteToP(fechaActual, P_ToP, V_ToP,
        V_Efectivo, fecha_ToP_sig, s, globs);
    end;
    //    //Dado que los embarques no son periódicos, debo buscar en el array de dias de llegada de embarques
    //    //el próximo embarque.
    //    diasSiguienteEmbarque := MaxInt;
    //    cntDias :=  Trunc(fechaActual) - Trunc(self.Fecha.dt);

    //    for i := 0 to length(DiasIngresoEmbarques) - 1 do
    //    begin
    //      diasAux := DiasIngresoEmbarques[i];
    //      if (diasAux < diasSiguienteEmbarque) and (diasAux >= cntDias) then
    //        diasSiguienteEmbarque := diasAux;
    //    end;
    //    fecha_ToP_sig := Trunc(self.Fecha.dt) + diasSiguienteEmbarque;

    //  end;
    //  result := fecha_ToP_sig;
  end;

end;

function TFichaSuministroCombustible_TakeOrPay.infoAd_: string;
begin
  Result := inherited  infoAd_;
  Result := 'T_ToP= ' + IntToStr(T_ToP) + 'dias' + 'V_ToP= ' +
    FloatToStrF(V_ToP, ffGeneral, 10, 1) + ' ' + combustible.Unidades +
    ', ' + 'P_ToP= ' + FloatToStrF(P_ToP, ffGeneral, 10, 1) +
    ' USD/' + combustible.Unidades + ', ' + 'P_ExtraToP= ' +
    FloatToStrF(P_ExtraToP, ffGeneral, 10, 2) + ' USD/' + combustible.Unidades + ', ';
end;

procedure TFichaSuministroCombustible_TakeOrPay.Free;
begin
  inherited Free;
end;




























procedure TFichaSuministroCombustible_TakeOrPay.AfterInstantiation;
begin
  inherited AfterInstantiation;
  self.agenda := TAgendaEmbarque.Create(capa, fecha, self.T_ToP,
    self.V_ToP, self.P_ToP, self.EmbarquesPeriodicos, strDiasIngresoEmbarques);
end;

//------------------------------
// Métodos de TSuministroCombustible_TakeOrPay
//==============================

constructor TSuministroCombustible_TakeOrPay.Create(capa: integer;
  nombre: string; nombreCombustible: string; nacimiento, muerte: TFecha;
  lpdUnidades, lpd: TFichasLPD; VDisp_ToP_Ini: NReal; NDisc: NInt);
begin
  inherited Create(capa, nombre, nombreCombustible, nacimiento, muerte,
    lpdUnidades, lpd);
  self.VDisp_ToP_Ini := VDisp_ToP_Ini;
  self.NDisc := NDisc;
end;


 
constructor TSuministroCombustible_TakeOrPay.Create_ReadFromText(f: TArchiTexto);
begin
  inherited Create_ReadFromText(f);
  f.IniciarLecturaRetrasada;
  f.rd('VDisp_ToP', VDisp_ToP_Ini);
  f.rd('NDisc', NDisc);
  f.EjecutarLectura;
  pa := nil;
end;



procedure TSuministroCombustible_TakeOrPay.PrepararMemoria(globs: TGlobs);
begin
  inherited prepararMemoria(globs);
end;

function TSuministroCombustible_TakeOrPay.get_ires_volumen: NInt;
begin
  Result := self.ires;
end;

function TSuministroCombustible_TakeOrPay.get_precio_combustible: NReal;
begin
  if (VConsumido_ToP > 0) and (VConsumido_Extra_Top = 0) then
    Result := dCFdV
  else
  if (VConsumido_ToP > 0) and (VConsumido_Extra_Top > 0) then
    Result := pa.P_ExtraToP
  else
  if (VConsumido_ToP = 0) and (VConsumido_Extra_Top > 0) then
    Result := (dCFdV * VConsumido_ToP + pa.P_ExtraToP * VConsumido_Extra_Top) /
      (VConsumido_ToP + VConsumido_Extra_Top);
end;


function TSuministroCombustible_TakeOrPay.get_combustible: TCombustibleSGE;
begin
  Result := self.combustible;
end;

function TSuministroCombustible_TakeOrPay.restricciones_postizadas: boolean;
begin
  Result := ((pa.C_Max <> -1) and pa.limites_postizados);
end;

procedure TSuministroCombustible_TakeOrPay.RegistrarParametrosDinamicos;
var
  i: integer;
  ficha: TFichaSuministroCombustible_TakeOrPay;
begin
  inherited registrarParametrosDinamicos;
  lpd.expandirFichas(globs);
  lpd.RegistrarFichasAActualizar(Self, globs.ActualizadorLPD, @pA, nil,
    TSuministroCombustible_TakeOrPay_cambioFichaPD);
  for i := 0 to lpd.Count - 1 do
  begin
    ficha := TFichaSuministroCombustible_TakeOrPay(lpd[i]);
    if ficha.indicePreciosPorCombustible <> nil then
      ficha.nroBornePreciosPorCombustible :=
        ficha.indicePreciosPorCombustible.IdBorne(ficha.bornePreciosPorCombustible);
    if ficha.agPerturbacionDias <> nil then
    begin
      ficha.agNroBornePerturbacionDias :=
        ficha.agPerturbacionDias.IdBorne(ficha.agBornePerturbacionDias);
      ficha.agenda.registrarPerturbacionDias(ficha.agPerturbacionDias,
        ficha.agNroBornePerturbacionDias);
    end;

  end;
end;

 
procedure TSuministroCombustible_TakeOrPay.WriteToText(f: TArchiTexto);
begin
  inherited WriteToText(f);
  f.wr('VDisp_ToP', VDisp_ToP_Ini);
  f.wr('NDisc', NDisc);
end;



procedure TSuministroCombustible_TakeOrPay.Free;
begin
  inherited Free;
end;

class function TSuministroCombustible_TakeOrPay.DescClase: string;
begin
  Result := 'Contrato Take Or Pay con Pago al Inicio del Período';
end;

procedure TSuministroCombustible_TakeOrPay.SorteosDelPaso(sortear: boolean);
var
  fecha_ToP_ant: TDateTime;  //Fecha del ultimo embarque recibido (anterior al paso)
  fecha_ToP_sig: TDateTime;  //Fecha del siguiente embarque
  i, cntDias: integer;
  ficha: TFichaSuministroCombustible_TakeOrPay;
  fichaActiva: TFichaSuministroCombustible_TakeOrPay;

  ptop: NReal;

begin
  V_ToP_sig := 0;

  fichaActiva := pa;

  //  fecha_ToP_sig := pa.GetFechaSiguienteToP(globs.FechaInicioDelpaso.dt);
  pa.GetFechaSiguienteToP(globs.FechaInicioDelpaso.dt, fecha_ToP_sig, globs);
  V_ToP_sig := pa.V_ToP;

  //Ahora debemos detectar si hay una ficha que comienza a regir en el paso de simulación.
  //En este caso, en el paso no se actualiza el volumen, se consume todo lo que resta
  //y el volumen se actualizará al final del paso de tiempo
  for i := 0 to lpd.Count - 1 do
  begin
    ficha := TFichaSuministroCombustible_TakeOrPay(lpd[i]);
    if (ficha.fecha.dt > globs.FechaInicioDelpaso.dt) and
      (ficha.fecha.dt < globs.FechaFinDelpaso.dt) then
      if (fecha_ToP_sig > ficha.fecha.dt) then
      begin
        if self.EmbarquesPeriodicos then
        begin
          fecha_ToP_sig := ficha.fecha.dt;
          V_ToP_sig := ficha.V_ToP;
        end;
        fichaActiva := ficha;
      end;
  end;

  //Ahora determinamos la fase en función de cuando ingreso el cargamento anterior y cuando ingresa el siguiente
  // no utilizo globs.FechaInicioDelpaso.dt porque tiene una hora menos!!!!!
  if (globs.FechaInicioDelpaso.dt <= fecha_ToP_sig) and
    ((globs.FechaInicioDelpaso.dt + globs.dt_DelPaso) > fecha_ToP_sig) then
  begin
    //Atención: se pone globs.FechaFinDelPaso.dt > fecha_ToP_sig pues en caso de la igualdad
    //se toma en el inicio del siguiente paso.
    fase := 1;     //En este paso de simulación ingresa un volumen T_ToP del contrato
    pa := fichaActiva;
  end
  else
    fase := 0;     //En este paso de simulación no ingresa combustible

  //Si estamos en la fase 1, debemos actualizar el volumen disponible al final del paso
  //y el volumen restante lo ponemos como perdido
  //Si estamos en la fase 0, significa que tenemos un contrato vigente y estamos a la espera de un cargamento ToP

end;


procedure TSuministroCombustible_TakeOrPay.PrepararPaso_ps;
var
  dCFdV_Inc, dCFdV_Dec: NReal;
  rescod: integer;
  xrpos: double;
begin

  self.VConsumido_ToP := 0;
  self.VConsumido_Extra_Top := 0;
  self.VSobrante_ToP := 0;
  //Los suministros de combustible se preparan antes que los generadores. Esto permite que
  //el generador tenga las cosas que necesita del suministro de combustible.

  cvToP := 0;
  cvExtraToP := 0;

  if (pa.T_ToP = -1) then  //no va a recibir cargamentos o sea el tanque no juega
    cvToP := 0
  else
  begin
    globs.CF.devxr_continuo(ixr, globs.kPaso_ + 1, dCFdV_Inc,
      dCFdV_Dec, rescod, xrpos);

    dCFdV := dCFdV_Dec;
    cvToP := -dCFdV;
  end;

  if fase = 1 then
  begin
    if pa.indicePreciosPorCombustible <> nil then
    begin
      cfToP := pa.P_ToP * pa.V_ToP *
        pa.indicePreciosPorCombustible.bornera[pa.nroBornePreciosPorCombustible];
      cvExtraToP := pa.P_ExtraToP *
        pa.indicePreciosPorCombustible.bornera[pa.nroBornePreciosPorCombustible];
    end
    else
    begin
      cfToP := pa.P_ToP * pa.V_ToP;
      cvExtraToP := pa.P_ExtraToP;
    end;
  end
  else
  begin
    cfToP := 0;         //No ingresa combustible en esta etapa
    if pa.indicePreciosPorCombustible <> nil then
    begin
      cvExtraToP := pa.P_ExtraToP *
        pa.indicePreciosPorCombustible.bornera[pa.nroBornePreciosPorCombustible];
    end
    else
    begin
      cvExtraToP := pa.P_ExtraToP;
    end;
  end;

end;

procedure TSuministroCombustible_TakeOrPay.opt_nvers(var ivar, ivae, ires: integer);
begin
  self.ivar := ivar; //CAntidad de variables (columnas)
  self.ires := ires;  //Cantidad de restricciones (filas)

  if pa.C_Max <> -1 then
  begin
    if pa.limites_postizados then
    begin
      ivar := ivar + globs.NPostes;  // Columnas de VConsumido x poste, VExtraTop y VToP
      ires := ires + globs.NPostes;
      // 1 restricción por poste. Luego los generadores deberán completar en sus columnas y en cada fila la restricción de volumen
    end
    else
    begin
      ivar := ivar + 1;
      ires := ires + 1;
    end;
  end;

  indice_var_ETop := ivar;
  ivar := ivar + 2; // Columnas de VConsumido x poste, VExtraTop y VToP
  indice_res_ETop := ires;
  ires := ires + 1;
  // 1 restricción por poste. Luego los generadores deberán completar en sus columnas y en cada fila la restricción de volumen
end;

procedure TSuministroCombustible_TakeOrPay.opt_cargue(s: TSimplex);
var
  i: integer;

begin
{$IFDEF SPXCONLOG}
  spx_NombrarVariables(s);
{$ENDIF}


  if pa.C_Max <> -1 then
  begin
    if pa.limites_postizados then
    begin
      //Restricción de balance de volumen (El volumen consumido en cada poste - VConsumido debe ser igual a 0)
      for i := 0 to globs.NPostes - 1 do
      begin
        s.pon_e(ires + i, ivar + i, -1);
        //Restriccion de balance de volumen por poste con VTop y VExtraTop
        s.pon_e(ires + globs.NPostes, ivar + i, -1);
      end;

    end
    else
    begin
      s.pon_e(ires, ivar, -1);
      s.pon_e(ires + 1, ivar, -1);
    end;
    s.pon_e(indice_res_ETop, indice_var_ETop, 1);
    s.pon_e(indice_res_ETop, indice_var_ETop + 1, 1);
  end
  else
  begin
    s.pon_e(indice_res_ETop, indice_var_ETop, -1);
    s.pon_e(indice_res_ETop, indice_var_ETop + 1, -1);
  end;



  //VTop <= VDisponible
  //s.pon_e(ires + (globs.NPostes * 2) + 1, ivar + (globs.NPostes * 2) + 1, 1);

  //Aporte a la función de Costo
  s.pon_e(s.nf, indice_var_ETop, -cvExtraToP);
  s.pon_e(s.nf, indice_var_ETop + 1, -cvToP);

  //Termino independiente (vale o o Vtop*Ptop)
  s.acum_e(s.nf, s.nc, -cfToP);

  //Cada uno de los generadores conectados debe ser responsable de cargar sus
  //columnas en la restricción de volumen  y en la función de costo objetivo
end;

procedure TSuministroCombustible_TakeOrPay.Sim_Cronica_Inicio;
var
  f: integer;
  ficha: TFichaSuministroCombustible_TakeOrPay;
begin
  inherited Sim_Cronica_Inicio;
  X_VDisp_ToP := VDisp_ToP_Ini;   //Inicializo la variable de estado.
  for  f := 0 to lpd.Count - 1 do
  begin
    ficha := TFichaSuministroCombustible_TakeOrPay(lpd.f[f]);
    ficha.agenda.perturbacionAgenda();
  end;
end;

procedure TSuministroCombustible_TakeOrPay.opt_fijarRestriccionesDeCaja(s: TSimplex);
var
  iposte: integer;
  i: integer;
begin

  if pa.C_Max <> -1 then
  begin
    if pa.limites_postizados then
    begin
      for i := 0 to globs.NPostes - 1 do
      begin
        s.FijarRestriccionIgualdad(ires + i);
        //Restricción de caudal máximo por poste
        s.cota_sup_set(ivar + i, pa.C_Max * globs.DurPos[i]);
      end;
      s.FijarRestriccionIgualdad(ires + globs.NPostes);
    end
    else
    begin
      //Restricción de caudal máximo por poste
      s.cota_sup_set(ivar, pa.C_Max * globs.HorasDelPaso);
      s.FijarRestriccionIgualdad(ires);
      s.FijarRestriccionIgualdad(ires + 1);
    end;
  end
  else
    s.FijarRestriccionIgualdad(ires);


  if (pa.T_ToP = -1) then  //no va a recibir cargamentos o sea el tanque no juega
    //Agregamos la restricción de caja que dice que el  0 <= VConsumidoToP >= 0
    s.cota_sup_set(indice_var_ETop + 1, 0)
  else
    //Agregamos la restricción de caja que dice que el  0 <= VConsumidoToP >= VDisponibleTop
    s.cota_sup_set(indice_var_ETop + 1, X_VDisp_ToP);

end;

procedure TSuministroCombustible_TakeOrPay.opt_leerSolucion(s: TSimplex);
var
  VDisp_ToP_: NReal;

  BOG_T_aux, BOG_GSS_aux, BOG_RG_aux, BOG_RC_aux, BOG_RL_aux,
  BOG_GRL_aux, BOG_Vent_aux, BSO_aux: NReal;
  iposte: integer;

begin
  self.VConsumido_Extra_Top := s.xval(indice_var_ETop);
  self.VConsumido_ToP := s.xval(indice_var_ETop + 1);
  //Calculo del costo directo del paso y actualizo el estado al final del paso

  //Costo Fijo mas variable. El fijo es 0 si no estoy fecha de renovación de contrato, o VTop*PTop
  //y cvExtraTop es el Volumen Extra por el Precio extra Top (esto ya se calculo al preparar el paso)

  costoTop := cfToP;
  costoExtraTop := cvExtraToP * self.VConsumido_Extra_Top;
  CostoDirectoDelPaso := cfToP + cvExtraToP * self.VConsumido_Extra_Top;


  VSobrante_ToP := 0;
  if fase = 1 then
  begin
    VSobrante_ToP := X_VDisp_ToP - self.VConsumido_ToP;
    //Lo que quedó disponible en el tanque es sobrante (sin consumir)
    Xs_VDisp_ToP := V_ToP_sig;       //Actualizo el estado, ingreso el combustible

    //CostoDirectoDelPaso := CostoDirectoDelPaso +  VSobrante_ToP* (cfToP/V_ToP_sig);

  end
  else
  begin
    Xs_VDisp_ToP := X_VDisp_ToP - self.VConsumido_ToP;
    //En esta fase, simplemente consumí y actualize el estado
    if Xs_VDisp_ToP < 0 then
      Xs_VDisp_ToP := 0;
  end;

  BOG_T := 0;
  BOG_GSS := 0;
  BOG_RG := 0;
  BOG_RC := 0;
  BOG_RL := 0;
  BOG_GRL := 0;
  BOG_Vent := 0;
  BSO := 0;

  if pa.manejoBOG then
  begin
    if pa.limites_postizados then
    begin
      for iposte := 0 to globs.NPostes - 1 do
      begin
        BOG_Calc.BOG_poste(BOG_T_aux, BOG_GSS_aux, BOG_RG_aux,
          BOG_RC_aux, BOG_RL_aux, BOG_GRL_aux, BOG_Vent_aux, BSO_aux,
          s.xval(ivar + iposte), X_VDisp_ToP, pa.K_BOG, pa.K_GSS,
          pa.K_RG, pa.K_RC, pa.K_RL, pa.Relicuado, globs.DurPos[iposte]);

        //        BOG_Calc.BOG_poste(BOG_T_aux, BOG_GSS_aux, BOG_RG_aux, BOG_RC_aux, BOG_RL_aux, BOG_GRL_aux, BOG_Vent_aux, BSO_aux,
        //        s.xval(ivar + iposte), X_VDisp_ToP, 0, 0, 0, 0, 0, false, globs.DurPos[iposte+1]);

        BOG_T := BOG_T + BOG_T_aux;
        BOG_GSS := BOG_GSS + BOG_GSS_aux;
        BOG_RG := BOG_RG + BOG_RG_aux;
        BOG_RC := BOG_RC + BOG_RC_aux;
        BOG_RL := BOG_RL + BOG_RL_aux;
        BOG_GRL := BOG_GRL + BOG_GRL_aux;
        BOG_Vent := BOG_Vent + BOG_Vent_aux;
        BSO := BSO + BSO_aux;

      end;

    end
    else
    begin
      BOG_Calc.BOG_poste(BOG_T_aux, BOG_GSS_aux, BOG_RG_aux,
        BOG_RC_aux, BOG_RL_aux, BOG_GRL_aux, BOG_Vent_aux, BSO_aux,
        //        s.xval(ivar + iposte), X_VDisp_ToP, 0.01/24, 0, 0.2/24, 0.1/24, 1/12, true, globs.DurPos[iposte+1]);
        self.VConsumido_ToP, X_VDisp_ToP, pa.K_BOG, pa.K_GSS, pa.K_RG,
        pa.K_RC, pa.K_RL, pa.Relicuado, globs.HorasDelPaso);

      //        BOG_Calc.BOG_poste(BOG_T_aux, BOG_GSS_aux, BOG_RG_aux, BOG_RC_aux, BOG_RL_aux, BOG_GRL_aux, BOG_Vent_aux, BSO_aux,
      //        self.VConsumido_ToP, X_VDisp_ToP, 0, 0, 0, 0, 0, false, globs.HorasDelPaso);

      BOG_T := BOG_T + BOG_T_aux;
      BOG_GSS := BOG_GSS + BOG_GSS_aux;
      BOG_RG := BOG_RG + BOG_RG_aux;
      BOG_RC := BOG_RC + BOG_RC_aux;
      BOG_RL := BOG_RL + BOG_RL_aux;
      BOG_GRL := BOG_GRL + BOG_GRL_aux;
      BOG_Vent := BOG_Vent + BOG_Vent_aux;
      BSO := BSO + BSO_aux;
    end;
  end;




  Xs_VDisp_ToP := Xs_VDisp_ToP - BOG_T + BOG_RC + BOG_RL - BSO;

end;

procedure TSuministroCombustible_TakeOrPay.EvolucionarEstado;
begin
  if (pa.T_ToP = -1) then
    exit; //no va a recibir cargamentos o sea el tanque no juega

  X_VDisp_ToP := max(Xs_VDisp_ToP, 0);
end;

function TSuministroCombustible_TakeOrPay.getNombreVar(ivar: integer;
  var nombre: string): boolean;
var
  iposte: integer;
begin

  if ((ivar >= self.ivar) and (ivar <= self.ivar + globs.NPostes - 1)) then
  begin
    iposte := ivar - self.ivar;
    nombre := self.Nombre + '_VCons_' + IntToStr(iposte + 1);
    Result := True;
  end
  else

  if (ivar = self.ivar + globs.NPostes) then
  begin
    nombre := self.Nombre + '_V_Extra_ToP[' + pa.combustible.Unidades + ']';
    Result := True;
  end
  else
  if (ivar = self.ivar + globs.NPostes + 1) then
  begin
    nombre := self.Nombre + '_V_Consumido_ToP[' + pa.combustible.Unidades + ']';
    Result := True;
  end
  else
    Result := False;
end;

function TSuministroCombustible_TakeOrPay.getNombreRes(ires: integer;
  var nombre: string): boolean;
var
  iposte: integer;
begin
  if ((ires >= self.ires) and (ires <= self.ires + globs.NPostes - 1)) then
  begin
    iposte := ires - self.ires;
    nombre := self.Nombre + '_Consumido_' + IntToStr(iposte + 1);
    Result := True;
  end
  else
  if ires = self.ires + globs.NPostes then
  begin
    nombre := self.Nombre + '_Consumido_Total';
    Result := True;
  end
  else
    Result := False;
end;

procedure TSuministroCombustible_TakeOrPay.PubliVars;
var
  unidades: string;
  ficha: TFichaSuministroCombustible_TakeOrPay;
  i: NInt;
begin
  inherited PubliVars;

  unidades := self.combustible.Unidades;

  PublicarVariableNR('VConsumido_ToP', 'm3', 6, 1, VConsumido_ToP, True);
  PublicarVariableNR('VConsumido_Extra_Top', 'm3', 6, 1, VConsumido_Extra_Top, True);
  PublicarVariableNR('VDisponible_ToP', 'm3', 6, 1, X_VDisp_ToP, True);
  PublicarVariableNR('VSobrante_ToP', 'm3', 6, 1, VSobrante_ToP, True);
  PublicarVariableNR('BOG_T', 'm3', 6, 1, BOG_T, True);
  PublicarVariableNR('BOG_GSS', 'm3', 6, 1, BOG_GSS, True);
  PublicarVariableNR('BOG_RG', 'm3', 6, 1, BOG_RG, True);
  PublicarVariableNR('BOG_RC', 'm3', 6, 1, BOG_RC, True);
  PublicarVariableNR('BOG_RL', 'm3', 6, 1, BOG_RL, True);
  PublicarVariableNR('BOG_GRL', 'm3', 6, 1, BOG_GRL, True);
  PublicarVariableNR('BOG_Vent', 'm3', 6, 1, BOG_Vent, True);
  PublicarVariableNR('BSO', 'm3', 6, 1, BSO, True);
  PublicarVariableNR('CostoVariableCombustible', 'USD/m3', 6, 1, cvToP, True);
  PublicarVariableNR('CostoToP', 'USD', 6, 1, costoToP, True);
  PublicarVariableNR('CostoExtraToP', 'USD', 6, 1, costoExtraToP, True);
  PublicarVariableNR('CostoDirectoDelPaso', 'USD', 6, 1, costoDirectoDelPaso, True);
  PublicarVariableNI('Fase', '-', fase, True);

end;















procedure TSuministroCombustible_TakeOrPay.AfterInstantiation;
begin
  inherited AfterInstantiation;
  pa := nil;
end;

{
function TSuministroCombustible_TakeOrPay.varsPSimRes3PorDefecto: TStringList;
var
  res: TStringList;
begin
{  SetLength(res, 6);

  res[0]:= 'Costo_Directo_Paso';
  res[1]:= 'CV_combustible';
  res[2]:= 'V_Consumido_ToP';
  res[3]:= 'V_Consumido_Extra_ToP';
  res[4]:= 'V_Sobrante_ToP';
  res[5]:= 'V_Disponible_ToP';
  res[6]:= 'Fase';
}
{
  res:= inherited varsPSimRes3PorDefecto;
  res.Add( 'Costo_Directo_Paso' );
  res.Add( 'CV_combustible' );
  res.Add( 'V_Consumido_ToP' );
  res.Add( 'V_Consumido_Extra_ToP' );
  res.Add( 'V_Sobrante_ToP' );
  res.Add( 'V_Disponible_ToP' );
  res.Add( 'Fase' );

  result:= res;
end;
}

procedure TSuministroCombustible_TakeOrPay.optx_nvxs(
  var ixr, ixd, iauxr, iauxd: integer);
begin
  if (pa.T_ToP = -1) then
    exit; //no va a recibir cargamentos o sea el tanque no juega
  self.ixr := ixr;
  ixr := ixr + 1;
end;

procedure TSuministroCombustible_TakeOrPay.PosicionarseEnEstrellita;
begin
  if (pa.T_ToP = -1) then
    exit; //no va a recibir cargamentos o sea el tanque no juega
  X_VDisp_ToP := globs.CF.xr[ixr];
end;

procedure TSuministroCombustible_TakeOrPay.ActualizarEstadoGlobal( flg_Xs: boolean );
begin

// OJO mirar pertinencia de usar flg_Xs

  if (pa.T_ToP = -1) then
    exit; //no va a recibir cargamentos o sea el tanque no juega

  globs.CF.xr[ixr] := X_VDisp_ToP;
end;

procedure TSuministroCombustible_TakeOrPay.optx_RegistrarVariablesDeEstado(
  adminEstados: TAdminEstados);
begin
  if (pa.T_ToP = -1) then
    exit; //no va a recibir cargamentos o sea el tanque no juega

  adminEstados.Registrar_Continua(
    ixr,
    0.0,
    pa.V_ToP,
    NDisc,
    nombre + '_V_Disponible_ToP', // nombre de la variable
    combustible.Unidades // unidades
    );
end;


procedure TSuministroCombustible_TakeOrPay.opt_PrintResultados_Encab(
  var fsal: textfile);
begin
  Write(fsal, #9, FloatToStrF(X_VDisp_ToP, ffgeneral, 6, 3));
end;

procedure TSuministroCombustible_TakeOrPay.opt_PrintResultados(var fsal: textfile);
begin
  Write(fsal, #9, FloatToStrF(dCFdV, ffgeneral, 6, 3));
end;

{
// imprime volumenes de combustible despachados, h_actual, cv_agua_Dec
procedure TSuministroCombustible_TakeOrPay.Sim_PrintResultados_Encab(var fsal: textfile; kencab: integer );
begin

  if kencab = 0 then
  begin
//    Write(fsal, #9,  Nombre, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre,#9, Nombre,#9, Nombre,#9, Nombre);
      write(fsal, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre,#9, Nombre, #9, Nombre, #9, Nombre, #9, Nombre);
  end
  else if kencab = 1 then
  begin
//    write(fsal, #9, '[VConsumido_ToP]', #9, '[VConsumido_Extra_Top]', #9, '[VDisponible_ToP]', #9, '[VSobrante_ToP]', #9, '[CostoVariableCombustible]',#9, '[CostoToP]', #9, '[CostoExtraToP]', #9, '[Fase]');
    write(fsal, #9, '[VConsumido_ToP]', #9, '[VConsumido_Extra_Top]', #9, '[VDisponible_ToP]', #9, '[VSobrante_ToP]', #9, '[BOG_T]', #9, '[BOG_GSS]', #9, '[BOG_RG]', #9, '[BOG_RC]', #9, '[BOG_RL]', #9, '[BOG_GRL]', #9, '[BOG_Vent]', #9, '[BSO]', #9, '[CostoVariableCombustible]',#9, '[CostoToP]', #9, '[CostoExtraToP]', #9, '[CostoDirectoDelPaso]', #9, '[Fase]');
  end
  else if kencab = 2 then
  begin
//    write(fsal, #9, 'VConsumido_ToP', #9, 'VConsumido_Extra_Top', #9, 'VDisponible_ToP', #9, 'VSobrante_ToP', #9, 'CostoVariableCombustible', #9, 'CostoToP', #9, 'CostoExtraToP', #9, 'Fase');
    write(fsal, #9, 'VConsumido_ToP', #9, 'VConsumido_Extra_Top', #9, 'VDisponible_ToP', #9, 'VSobrante_ToP', #9,'BOG_T', #9, 'BOG_GSS', #9, 'BOG_RG', #9, 'BOG_RC', #9, 'BOG_RL', #9, 'BOG_GRL', #9, 'BOG_Vent', #9, 'BSO', #9, 'CostoVariableCombustible', #9, 'CostoToP', #9, 'CostoExtraToP', #9, 'CostoDirectoDelPaso', #9, 'Fase');
  end
  else
  begin
//    write(fsal, #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0');
write(fsal, #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0', #9, '0',#9, '0', #9, '0', #9, '0', #9, '0');
  end;
end;

procedure TSuministroCombustible_TakeOrPay.Sim_PrintResultados(var fsal: textfile);

begin
   write( fsal, #9, FloatToStrF(VConsumido_ToP, formatoReales, 6, 1));
  write( fsal, #9, FloatToStrF(VConsumido_Extra_Top, formatoReales, 6, 1));
  write( fsal, #9, FloatToStrF(X_VDisp_ToP, formatoReales, 6, 1));
  write( fsal, #9, FloatToStrF(VSobrante_ToP, formatoReales, 6, 1));

  write( fsal, #9, FloatToStrF(BOG_T, formatoReales, 6, 1));
  write( fsal, #9, FloatToStrF(BOG_GSS, formatoReales, 6, 1));
  write( fsal, #9, FloatToStrF(BOG_RG, formatoReales, 6, 1));
  write( fsal, #9, FloatToStrF(BOG_RC, formatoReales, 6, 1));
  write( fsal, #9, FloatToStrF(BOG_RL, formatoReales, 6, 1));
  write( fsal, #9, FloatToStrF(BOG_GRL, formatoReales, 6, 1));
  write( fsal, #9, FloatToStrF(BOG_Vent, formatoReales, 6, 1));
  write( fsal, #9, FloatToStrF(BSO, formatoReales, 6, 1));


  write( fsal, #9, FloatToStrF(cvToP, formatoReales, 6, 6 ) );
  write( fsal, #9, FloatToStrF(costoToP, formatoReales, 6, 1 ) );
  write( fsal, #9, FloatToStrF(costoExtraToP, formatoReales, 6, 1 ) );
  write( fsal, #9, FloatToStrF(costoDirectoDelPaso, formatoReales, 6, 1 ) );
  write( fsal, #9, IntToStr(fase));

end;
}

procedure TSuministroCombustible_TakeOrPay_cambioFichaPD(Actor: TCosa);
begin
end;

procedure AlInicio;
begin
  registrarClaseDeCosa(TSuministroCombustible_TakeOrPay.ClassName,
    TSuministroCombustible_TakeOrPay);
  registrarClaseDeCosa(TFichaSuministroCombustible_TakeOrPay.ClassName,
    TFichaSuministroCombustible_TakeOrPay);
end;

procedure AlFinal;
begin
end;

end.
