{+doc
+NOMBRE:   uParqueEolico
+CREACION: 22/12/2006
+AUTORES:  rch
+REGISTRO:
+TIPO: Unidad Pascal.
+PROPOSITO:   Definición del actor TParqueEolico
+PROYECTO: SimSEE
+REVISION:
+AUTOR:
+DESCRIPCION:

-doc}

unit uParqueEolico;


interface

uses
  SysUtils,
  Classes,
  uFechas,
  uGeneradores,
  uFuentesAleatorias,
  uGlobs,
  uNodos,
  uCosa, uCosaConNombre,
  usimplex,
  uFichasLPD,
  uFuncionesReales,
  uConstantesSimSEE,
  xMatDefs;

resourcestring
  mesFuenteAleatConPaso1h =
    'TParqueEolico, necesita una fuente aleatoria con paso de sorteo = 1h. ';
  mesElParque = 'El parque ';
  mesConectadoAFuente = ' está conectado a la fuente ';
  mesConPasoDeSorteo = ' con paso de sorteo = ';
  rsParqueEolico = 'Parque eólico';
  exCantSpeedupParque = 'La cantidad de factores de speedup en el parque eolico ';
  exDebeSer12 = ' debe ser 12';

type

  { TParqueEolico }
  TParqueEolico = class(TGeneradorPostizador)
  public
  (**************************************************************************)
  (*              A T R I B U T O S   P E R S I S T E N T E S               *)
  (**************************************************************************)

    fdisp: NReal;
    fSpeedUpMes: TDAofNReal;
    fPerdidasInterferencias: NReal;
    CurvaVP: TFVectR;
    fuenteDeViento: TFuenteAleatoria;
    nombreBorne: string;

    PagoPorEnergiaDisponible_USD_MWh: NReal;
    tRepHoras: NReal;
    PagoPorEnergiaEntregada_USD_MWh: NReal; // [USD/MWh] para cálculo de CAD
  (**************************************************************************)

    //[USD/MWh] Pago por potencia disponible contemplando las unidades disponibles
    // resultado de los sorteos de disponibilidad y del mantenimiento programado y el viento

    kBorneVelocidad: integer;
    {$IFDEF CALC_DEMANDA_NETA}
    kBornePotencia: integer;
    {$ELSE}
    kBornePPA: TDAofNInt; // Potencia del Poste Actual
    {$ENDIF}

    fComplexivoMes: TDAofNReal;
    VVel: NReal;

    constructor Create(capa: integer; nombre: string; nacimiento, muerte: TFecha;
      lpdUnidades: TFichasLPD; nodo: TNodo;
  flg_CalcularGradienteDeInversion: boolean; xFDV: TFuenteAleatoria;
  xNombreBorne: string; xfdisp: NReal; tRepHoras: NReal;
  xfSpeedUpMes: TDAofNReal; xfPerdidasInterferencias: NReal;
  xCurvaVelocidadPotencia_Pot: TFVectR;
  xPagoPorEnergiaEntregada_USD_MWh: NReal;
  xPagoPorEnergiaDisponible_USD_MWh: NReal; xflg_RestarParaPostizado: boolean;
  TonCO2xMWh: NReal; LowCostMustRun, CleanDevelopmentMechanism: boolean;
  xFuenteIdxP: TFuenteAleatoria; xBorneIdxP: string);

    function Rec: TCosa_RecLnk; override;
    procedure BeforeRead(version, id_hilo: integer); override;
    procedure AfterRead(f:TArchiTexto); override;

    procedure PrepararMemoria( Catalogo: TCatalogoReferencias; globs: TGlobs); override;

    function PotenciaFirme: NReal; override;

    function InfoAd_: string; override;

    class function DescClase: string; override;

    function get_pa_FD(kTipoUnidad: integer): NReal; override;
    function get_pa_TMR(kTipoUnidad: integer): NReal; override;

    procedure SorteosDelPaso(sortear: boolean); override;
    procedure PrepararPaso_ps; override;

    procedure opt_nvers(var ivar, ivae, ires: integer); override;
    procedure opt_cargue(s: TSimplex); override;
    procedure opt_fijarRestriccionesDeCaja(s: TSimplex); override;
    procedure opt_leerSolucion(s: TSimplex); override;

    procedure Free; override;
    procedure PubliVars; override;

    procedure dump_Variables(var f: TextFile; charIndentacion: char); override;

    function getNombreVar(ivar: integer; var nombre: string): boolean; override;
    procedure CalcularPagoServicioDeConfiabilidaddelSist; override;

  private
    px_fSpeedUP: NReal;
  end;




procedure AlInicio;
procedure AlFinal;

implementation


constructor TParqueEolico.Create(capa: integer; nombre: string;
  nacimiento, muerte: TFecha; lpdUnidades: TFichasLPD; nodo: TNodo;
  flg_CalcularGradienteDeInversion: boolean;
  xFDV: TFuenteAleatoria; xNombreBorne: string;
  xfdisp: NReal;
  tRepHoras: NReal; xfSpeedUpMes: TDAofNReal; xfPerdidasInterferencias: NReal;
  xCurvaVelocidadPotencia_Pot: TFVectR; xPagoPorEnergiaEntregada_USD_MWh: NReal;
  xPagoPorEnergiaDisponible_USD_MWh: NReal; xflg_RestarParaPostizado: boolean;
  TonCO2xMWh: NReal; LowCostMustRun, CleanDevelopmentMechanism: boolean;
  xFuenteIdxP: TFuenteAleatoria; xBorneIdxP: string );
var
  i: integer;
begin
  inherited Create(capa, nombre, nacimiento, muerte, lpdUnidades, nodo,
    TonCO2xMWh, LowCostMustRun, CleanDevelopmentMechanism, xflg_RestarParaPostizado,
      flg_CalcularGradienteDeInversion,
      xFuenteIdxP, xBorneIdxP );

  fdisp := xfdisp;
  self.tRepHoras := tRepHoras;
  if length(xfSpeedUpMes) <> 12 then
    raise Exception.Create(exCantSpeedupParque + nombre + exDebeSer12);
  fSpeedUpMes := xfSpeedUpMes;
  fPerdidasInterferencias := xfPerdidasInterferencias;
  SetLength(fComplexivoMes, 12);
  for i := 0 to high(fSpeedUpMes) do
    fComplexivoMes[i] := fSpeedUpMes[i] * fPerdidasInterferencias;

  CurvaVP := xCurvaVelocidadPotencia_Pot;
  FuenteDeViento := xFDV;
  NombreBorne := xNombreBorne;
  kBorneVelocidad := -1;
  PagoPorEnergiaEntregada_USD_MWh := xPagoPorEnergiaEntregada_USD_MWh;
  PagoPorEnergiaDisponible_USD_MWh := xPagoPorEnergiaDisponible_USD_MWh;
end;

function TParqueEolico.Rec: TCosa_RecLnk;
begin
  Result:=inherited Rec;
  Result.addCampoDef('fdisp', fdisp, 0, 7 );
  Result.addCampoDef('fSpeedUp', px_fSpeedUp, 0, 7 );
  Result.addCampoDef('fPerdidasInterferencias', fPerdidasInterferencias, 0, 7 );
  Result.addCampoDef('CurvaVP', TCosa(CurvaVP), 0, 7 );
  Result.addCampoDef_ref('fuenteDeViento', TCosa(fuenteDeViento), Self, 0, 7 );
  Result.addCampoDef('nombreBorne', nombreBorne, 0, 7 );
  Result.addCampoDef('fdisp', fdisp, 7, 14 );
  Result.addCampoDef('fSpeedUpMes', fSpeedUPMes, 7, 14 );
  Result.addCampoDef('fPerdidasInterferencias', fPerdidasInterferencias, 7, 14 );
  Result.addCampoDef('CurvaVP', TCosa(CurvaVP), 7, 14 );
  Result.addCampoDef_ref('fuenteDeViento', TCosa(fuenteDeViento), Self, 7, 14 );
  Result.addCampoDef('nombreBorne', nombreBorne, 7, 14 );
  Result.addCampoDef('fdisp', fdisp, 14 );
  Result.addCampoDef('tRepHoras', tRepHoras, 14 );
  Result.addCampoDef('fSpeedUpMes', fSpeedUPMes, 14 );
  Result.addCampoDef('fPerdidasInterferencias', fPerdidasInterferencias, 14 );
  Result.addCampoDef('CurvaVP', TCosa(CurvaVP), 14 );
  Result.addCampoDef_ref('fuenteDeViento', TCosa(fuenteDeViento), Self, 14 );
  Result.addCampoDef('nombreBorne', nombreBorne, 14 );
  Result.addCampoDef('pagoPorEnergia', PagoPorEnergiaEntregada_USD_MWh, 36, 98 );
  Result.addCampoDef('pagoPorPotenciaDisp', PagoPorEnergiaDisponible_USD_MWh, 36, 98 );
  Result.addCampoDef('PagoPorDisponibilidad_USD_MWh', PagoPorEnergiaDisponible_USD_MWh, 98, 103 );
  Result.addCampoDef('PagoPorEnergia_USD_MWh', PagoPorEnergiaEntregada_USD_MWh, 98, 103 );
  Result.addCampoDef('PagoPorEnergiaDisponible_USD_MWh', PagoPorEnergiaDisponible_USD_MWh, 103 );
  Result.addCampoDef('PagoPorEnergiaEntregada_USD_MWh', PagoPorEnergiaEntregada_USD_MWh, 103 );
end;

procedure TParqueEolico.BeforeRead(version, id_hilo: integer);
begin
  inherited BeforeRead(version, id_hilo);
  PagoPorEnergiaDisponible_USD_MWh := 0;
  PagoPorEnergiaEntregada_USD_MWh := 0;
end;

procedure TParqueEolico.AfterRead(f:TArchiTexto);
var
  i: integer;
begin
  inherited AfterRead(f);
  if f.Version < 7 then
  begin
    SetLength(fSpeedUpMes, 12);
    SetLength(fComplexivoMes, 12);
    for i := 0 to high(fSpeedUpMes) do
    begin
      fSpeedUpMes[i] := px_fSpeedUp;
      fComplexivoMes[i] := fPerdidasInterferencias * px_fSpeedUP;
    end;
    kBorneVelocidad := -1;
    self.tRepHoras := 15 * 24;
  end
  else if f.Version < 14 then
  begin
    SetLength(fComplexivoMes, 12);
    for i := 0 to high(fSpeedUpMes) do
        fComplexivoMes[i] := fPerdidasInterferencias * fSpeedUPMes[i];
    kBorneVelocidad := -1;
    self.tRepHoras := 15 * 24;
  end
  else
  begin
    SetLength(fComplexivoMes, 12);
    for i := 0 to high(fSpeedUpMes) do
    fComplexivoMes[i] := fPerdidasInterferencias * fSpeedUPMes[i];
    kBorneVelocidad := -1;
  end;
end;


procedure TParqueEolico.Free;
begin
  CurvaVP.Free;

  {$IFNDEF CALC_DEMANDA_NETA}
  SetLength(kBornePPA, 0);
  {$ENDIF}
  SetLength(fSpeedUpMes, 0);
  SetLength(fComplexivoMes, 0);
  SetLength(PPA, 0);

  inherited Free;
end;


procedure TParqueEolico.PubliVars;
begin
  inherited PubliVars;
  PublicarVariableVR('PotenciaGenerable', '[MW]', 6, 2, PPA, True, True);
  PublicarVariableNR('VVel', '[m/s]', 6, 2, VVel, True);
  PublicarVariableNI('NMaqsDisponibles', '-', NMaquinasDisponibles, True);
  PublicarVariableNR('CostoDirectoDelPaso', '[USD]', 12, 2,
    self.costoDirectoDelPaso, True);
end;



procedure TParqueEolico.PrepararMemoria(Catalogo: TCatalogoReferencias;
  globs: TGlobs);
var
  iposte: integer;
  auxf: TFRenR;
begin
  inherited prepararMemoria( Catalogo, globs);
  setlength(PPA, globs.NPostes);

  if not ((fuenteDeViento.durPasoDeSorteoEnHoras = 1) or
    ((fuenteDeViento.durPasoDeSorteoEnHoras = 0) and (globs.HorasDelPaso = 1))) then
    raise Exception.Create(mesFuenteAleatConPaso1h + mesElParque +
      nombre + mesConectadoAFuente + fuenteDeViento.nombre +
      mesConPasoDeSorteo + IntToStr(fuenteDeViento.durPasoDeSorteoEnHoras));

  kBorneVelocidad := fuenteDeViento.idBorne(nombreBorne);
  {$IFDEF CALC_DEMANDA_NETA}
  auxf := TFf_xmult_conSelectorYBuffer.Create(capa, CurvaVP,
    fComplexivoMes, 1, @globs.MesInicioDelPaso, @PHoraria_PreSorteosPorUnidad, @globs.kSubPaso_);


  kBornePotencia := fuenteDeViento.registrarFuncion(auxf, nombreBorne, -1);

  // Atención, con esto inhibimos que la fuente haga el resumen y lo hace el generador.
  if fuenteDeViento.ResumirMaxVar( globs ) <> nil then
  begin
    fuenteDeViento.ResumirMaxVar( globs ).flg_ResumirBorneras:= false;
    flg_ResumirPromediando:= false;
  end
  else
    flg_ResumirPromediando:= true;;

  {$ELSE}
  SetLength(kBornePPA, globs.NPostes);
  for iposte := 0 to high(kBornePPA) do
  begin
     (*
     Para cada poste definimos un borne auxiliar que llevará la cuenta
     de la energía generada por el parque en ese poste y así puede calcular
     la potencia media equivalente del parque para ese poste.
     Se supone que la fuente de viento tiene paso de sorteo HORARIO.
     *)

    auxf := TFf_xmult_conselector.Create(capa, CurvaVP, fComplexivoMes,
      1, @globs.MesInicioDelPaso, iposte, @globs.kPosteHorasDelPaso,
      @globs.kSubPaso_);
    kBornePPA[iposte] := fuenteDeViento.registrarFuncion(auxf, nombreBorne, iposte);
  end;
  {$ENDIF}
end;


function TParqueEolico.PotenciaFirme: NReal;
begin
  Result := 0;
end;

function TParqueEolico.InfoAd_: string;
begin
  Result := '';
end;

class function TParqueEolico.DescClase: string;
begin
  Result := rsParqueEolico;
end;


function TParqueEolico.get_pa_FD(kTipoUnidad: integer): NReal;
begin
  Result := fdisp;
end;

function TParqueEolico.get_pa_TMR(kTipoUnidad: integer): NReal;
begin
  Result := tRepHoras;
end;


procedure TParqueEolico.SorteosDelPaso(sortear: boolean);
begin
  if globs.ObligarDisponibilidad_1_ then
    NMaquinasDisponibles := paUnidades.nUnidades_Operativas[0]
  else if (sortear) then
  begin
    ActualizarProbabilidadesReparacionYRotura_(fdisp, tRepHoras);
    NMaquinasDisponibles := Sorteos_RepRotUnidades;
  end
  else
    NMaquinasDisponibles := paUnidades.nUnidades_Operativas[0];
  inherited SorteosDelPaso( sortear );
 end;

procedure TParqueEolico.PrepararPaso_ps;
var
  iposte: integer;
  jHora: integer;
begin
  inherited prepararPaso_ps;
  VVel := Self.fuenteDeViento.bornera[Self.kBorneVelocidad] *
    fSpeedUpMes[globs.MesInicioDelPaso - 1];
  {$IFNDEF CALC_DEMANDA_NETA}
  for iposte := 0 to high(PPA) do
    PPA[iposte] := Self.fuenteDeViento.bornera[Self.kBornePPA[iposte]] *
      NMaquinasDisponibles;
  {$ENDIF}
end;


procedure TParqueEolico.opt_nvers(var ivar, ivae, ires: integer);
begin
  if NMaquinasDisponibles > 0 then
  begin
  self.ivar := ivar;
  self.ivae := ivae;
  self.ires := ires;
  ivar := ivar + globs.NPostes;
  end;
end;

function TParqueEolico.getNombreVar(ivar: integer; var nombre: string): boolean;
begin
  if NMaquinasDisponibles > 0 then
  begin
  if (ivar >= self.ivar) and (ivar < self.ivar + globs.NPostes) then
  begin
    nombre := self.Nombre + '_P[MW]' + IntToStr(ivar - self.ivar + 1);
    Result := True;
  end
  else
    Result := False;
  end
  else
    result:= false;
end;

procedure TParqueEolico.CalcularPagoServicioDeConfiabilidaddelSist;
var
   iPoste: integer;
   CmgMenosTecho, TechoMenosCV: NReal;
begin
    ParticipacionSCS:= 0;
    ForzamientoSCS:=0;
    for iPoste:= 0 to high( P )  do
    begin
       CmgMenosTecho:= nodo.cmarg[iposte] - globs.TechoDelSpot;
       if  CmgMenosTecho > 0  then
           ParticipacionSCS:= ParticipacionSCS + P[iposte]* globs.durpos[iposte] * CmgMenosTecho // al ltriangulito.
    end
end;


procedure TParqueEolico.opt_cargue(s: TSimplex);
var
  iposte: integer;
begin
  if NMaquinasDisponibles = 0 then
    exit;

  for iposte := 0 to globs.NPostes - 1 do
    s.pon_e(nodo.ires + iposte, ivar + iposte, 1);
end;

procedure TParqueEolico.opt_fijarRestriccionesDeCaja(s: TSimplex);
var
  iposte: integer;
begin
  if NMaquinasDisponibles = 0 then
    exit;
  // Le fijamos como cota superior PMax a la potencia en todos los postes
  for iposte := 0 to globs.NPostes - 1 do
    if PPA[iposte] > AsumaCero then
      s.cota_sup_set(ivar + iposte, PPA[iposte])
    else
      s.FijarVariable(ivar + iposte, 0);
end;

procedure TParqueEolico.opt_leerSolucion(s: TSimplex);
var
  iposte: integer;
  e, eacum, disp: NReal;
begin
  costoDirectoDelPaso := 0;
  if NMaquinasDisponibles = 0 then
  begin
    vclear( P );
    //vclear( cv_Spot );
    for iposte := 0 to globs.NPostes - 1 do
    begin
      Lambda_P[iPoste]:= 0;
      cv_Spot[iPoste]:=get_cv_falla; // para que en el ordenamiento quede último, con costo variable igual a la falla;
    end;
    Ingreso_PorDisponibilidad_ := 0;
    Ingreso_PorEnergia_ := 0;
    exit;
  end;

  // recuperamos los valores de Potencia despachada
  eacum := 0;
  disp := 0;
  for iposte := 0 to globs.NPostes - 1 do
  begin
    e := s.xval(ivar + iposte);
    P[iposte] := e;
    Lambda_P[iPoste]:= s.xmult( ivar + iposte ) / globs.DurPos[iposte];
    cv_Spot[iPoste]:= Nodo.cmarg[iposte] - Lambda_P[iPoste];
    eacum := eacum + e * globs.DurPos[iposte];
    disp := disp + PPA[iposte] * globs.DurPos[iposte];
  end;

  Ingreso_PorDisponibilidad_ := disp * PagoPorEnergiaDisponible_USD_MWh;
  Ingreso_PorEnergia_ := eacum * PagoPorEnergiaEntregada_USD_MWh;

end;


procedure TParqueEolico.dump_Variables(var f: TextFile; charIndentacion: char);
var
  i: integer;
begin
  inherited dump_Variables(f, charIndentacion);
  writeln(f, charIndentacion, 'VVel[m/s]= ', FloatToStrF(VVel, ffFixed, 10, 3));
  for i := 0 to high(PPA) do
    writeln(f, charIndentacion, 'PPA[', i, '][MW]: ',
      FloatToStrF(PPA[i], ffFixed, 10, 3));
  writeln(f);
  //  writeln(f, charIndentacion, 'PotMediaGenerable[MW]= ', FloatToStrF(PotMediaGenerable, ffFixed, 10, 3));
  writeln(f, charIndentacion, 'NMaquinasDisponibles= ', NMaquinasDisponibles);
  writeln(f);
end;

procedure AlInicio;
begin
  registrarClaseDeCosa(TParqueEolico.ClassName, TParqueEolico);
end;

procedure AlFinal;
begin
end;

end.
