unit uCombustible;

interface

uses
  SysUtils,
  ucosa,
  ucosaConNombre,
  ucosaparticipedemercado,
  uFichasLPD, xMatDefs, ufechas, uconstantesSimSEE, uglobs;

const
  J_por_kcal = 4186.8; // kcal (internacional)
  kcal_por_J = 1 / J_por_kcal;

  lt_por_m3 = 1000.0;
  m3_por_lt = 1.0 / lt_por_m3;

  J_por_BTU = 1055.05585262;
  BTU_por_J = 1.0 / J_por_BTU;

  J_por_MWh = 1.0E6 * 3600.0;
  MWh_por_J = 1.0 / J_por_MWh;

  J_por_MBTU = 1.0E6 * J_por_BTU;
  MBTU_por_J = 1.0 / J_por_MBTU;


  MWh_por_MBTU = MWh_por_J * J_por_MBTU;
  MBTU_por_MWh = 1.0 / MWh_por_MBTU;

  Ton_por_kg = 1000.0;
  kg_por_Ton = 1.0 / Ton_por_kg;

type

  { TFichaCombustible }

  TFichaCombustible = class(TFichaLPD)
  public
    // precio: NReal;
    ro: NReal; // Densidad kg/m3
    PCI: NReal;   // Poder Calorífico Superior J/kg
    PCS: NReal;   // Poder Calorífico Inferior J/kg

    constructor Create(capa: integer; fecha: TFecha; periodicidad: TPeriodicidad;
      ro, PCI, PCS: NReal);

    function Rec: TCosa_RecLnk; override;
    procedure BeforeRead(version, id_hilo: integer); override;
    procedure AfterRead(f:TArchiTexto); override;

    procedure generarLineaResumen(var archi: TextFile); override;
    function InfoAd_: string; override;
    procedure Free; override;
  end;

  TClaseDeCombustible = class of TCombustible;

  { TCombustible }

  TCombustible = class(TCosaParticipeDeMercado)
  public
    pa: TFichaCombustible;
    //  unidad: string;

    // Variables auxiliares calculadas con cambios de fichas dinámicas
    ro_PCI: NReal;  // densidad * PCI el resultado es [J/m3]
    MWh_por_m3: NReal; // [MWh/m3]
    MWh_por_Q1h_: NReal; // [MWh/m3 * (3600 s/h)]

    constructor Create(capa: integer; nombre: string; lpd: TFichasLPD);
    //  densidad: NReal; unidad: string); reintroduce;

    function Rec: TCosa_RecLnk; override;
    procedure BeforeRead(version, id_hilo: integer); override;
    procedure AfterRead(f:TArchiTexto); override;

    function InfoAd_: string; override;

    procedure RegistrarParametrosDinamicos(CatalogoReferencias: TCatalogoReferencias);
      override;

    procedure CambioFichaPD; override;
    class function DescClase: string; override;


    function precio_USD_por_MWh_a_PCI(precio_por_unidad: NReal;
      unidades_del_precio: string; flg_precio_a_PCS: boolean): NReal;

  end;

procedure TCombustible_cambioFichaPD(Actor: TCosa);


procedure AlInicio;
procedure AlFinal;


resourcestring
  rsCombusitbles = 'Combustibles';


implementation

procedure TCombustible_cambioFichaPD(Actor: TCosa);
begin
  (Actor as TCombustible).cambioFichaPD;
end;



//------------------------------
// Métodos de TFichaCombustible
//==============================
constructor TFichaCombustible.Create(capa: integer; fecha: TFecha;
  periodicidad: TPeriodicidad; ro, PCI, PCS: NReal);
begin
  inherited Create(capa, fecha, periodicidad);
  self.ro := ro;
  self.PCI := PCI;
  self.PCS := PCS;
end;

function TFichaCombustible.Rec: TCosa_RecLnk;
begin
  Result:=inherited Rec;
  Result.addCampoDef('densidad', ro, 0, 114);
  Result.addCampoDef('ro', ro, 114);
  Result.addCampoDef('PCI', PCI);
  Result.addCampoDef('PCS', PCS);
end;

procedure TFichaCombustible.BeforeRead(version, id_hilo: integer);
begin
  inherited BeforeRead(version, id_hilo);
end;

procedure TFichaCombustible.AfterRead(f:TArchiTexto);
begin
  inherited AfterRead(f);
end;


procedure TFichaCombustible.generarLineaResumen(var archi: TextFile);
begin
  Write(archi, FloatToStrF(ro, formatoReales, 8, 1), #9,
    FloatToStrF(PCI, formatoReales, 8, 1), #9,
    FloatToStrF(PCS, formatoReales, 8, 2), #9);
end;

function TFichaCombustible.InfoAd_: string;
begin
  Result := 'ro[kg/m3]= ' + FloatToStrF(ro, ffGeneral, 10, 1) +
    ', PCI[J/kg]= ' + FloatToStrF(PCI, ffGeneral, 10, 1) + ', PCS[J/kg]= ' +
    FloatToStrF(PCS, ffGeneral, 10, 1);
end;

procedure TFichaCombustible.Free;
begin
  inherited Free;
end;



//------------------------------
// Métodos de TCombustible
//==============================

constructor TCombustible.Create(capa: integer; nombre: string; lpd: TFichasLPD);
begin
  inherited Create(capa, nombre);
  self.lpd := lpd;
end;

function TCombustible.Rec: TCosa_RecLnk;
begin
  Result:=inherited Rec;
  Result.addCampoDef('lpd', TCosa(lpd));
end;

procedure TCombustible.BeforeRead(version, id_hilo: integer);
begin
  inherited BeforeRead(version, id_hilo);
end;

procedure TCombustible.AfterRead(f:TArchiTexto);
begin
  inherited AfterRead(f);
  lpd.Propietario:= self;
end;

function TCombustible.InfoAd_: string;
begin
  Result := '';
end;

procedure TCombustible.RegistrarParametrosDinamicos(
  CatalogoReferencias: TCatalogoReferencias);
begin
  lpd.expandirFichas(CatalogoReferencias, globs);
  lpd.RegistrarFichasAActualizar(Self, globs.ActualizadorLPD, @pA,
    nil, TCombustible_cambioFichaPD);

  // Atención aquí hago esto para que el Combustible tenga definida pA por lo menos
  // en el arranque con la primer ficha. Esto es necesario pues el TGSimple_bicombustible
  // usa los parámetros de Combustible en cambio_ficha_lpd
  pA := lpd.items[0] as TFichaCombustible;
  CambioFichaPD;
end;

procedure TCombustible.CambioFichaPD;
begin
  ro_PCI := pa.ro * pa.PCI; // J/m3
  MWh_por_m3 := ro_PCI * MWh_por_J;
  MWh_por_Q1h_ := MWh_por_m3 * 3600;
end;


class function TCombustible.DescClase: string;
begin
  Result := rsCombusitbles;
end;

function TCombustible.precio_USD_por_MWh_a_PCI(precio_por_unidad: NReal;
  unidades_del_precio: string; flg_precio_a_PCS: boolean): NReal;
var
  res: NReal;
begin
  if unidades_del_precio = 'USD/MBTU' then
    res := precio_por_unidad * MBTU_por_MWh
  else if unidades_del_precio = 'USD/m3' then
    res := precio_por_unidad / (pa.ro * pa.PCS * MWh_por_J)
  else if unidades_del_precio = 'USD/Ton' then
    res := precio_por_unidad / (Ton_por_kg * pa.PCS * MWh_por_J)
  else if unidades_del_precio = 'USD/kg' then
    res := precio_por_unidad / (pa.PCS * MWh_por_J)
  else if unidades_del_precio = 'USD/MWh' then
    res := precio_por_unidad;

  if flg_precio_a_PCS then
    res := res * pa.PCS / pa.PCI;

  Result := res;
end;

procedure AlInicio;
begin
  registrarClaseDeCosa(TCombustible.ClassName, TCombustible);
  registrarClaseDeCosa(TFichaCombustible.ClassName, TFichaCombustible);
end;

procedure AlFinal;
begin
end;

end.
