unit ugter_arranqueparada_noopt;

interface

uses
  SysUtils, Classes, xmatdefs, uGTer, unodos,
  uglobs,
  umipsimplex,
  usimplex,
  ufichasLPD,
  ufechas,
  ucosa, uCosaConNombre,
  uconstantesSimSEE,
  uFuentesAleatorias;

resourcestring
  rsGeneradorTermicoConCostoArranqueParada = 'GTer con costos arranque/parada';

type

  { TFichaGTer_ArranqueParada_noopt }
  TFichaGTer_ArranqueParada_noopt = class(TFichaLPD)

  public

    (**************************************************************************)
    (*               A T R I B U T O S   P E R S I S T E N T E S              *)
    (**************************************************************************)

    PMin, PMax: NReal; // [MW] Potencias Mínima y Máxima Por maquina
    cv_min, cv: NReal; // Costo:= cv_min* Pmin+ cv* (P-Pmin)
    indicePreciosPorCombustible: TFuenteAleatoria;
    bornePreciosPorCombustible: string;
    disp: NReal; // disponibilidad (fortuita)
    tRepHoras: NReal;
    costo_arranque, costo_parada: NReal;
    HayRestriccionEmaxPasoDeTiempo: boolean; // indica si se aplica la restricción
    EmaxPasoDeTiempo: NReal; // Energía maxima generable en un paso de tiempo

    (**************************************************************************)

    nroBornePreciosPorCombustible: integer;

    constructor Create(capa: integer; fecha: TFecha; periodicidad: TPeriodicidad;
      PMin, PMax: NReal; cv_min, cv: NReal;
      indicePreciosPorCombustible: TFuenteAleatoria; bornePreciosPorCombustible: string;
      disp: NReal; tRepHoras: NReal; costo_arranque, costo_parada: Nreal;
      HayRestriccionEmaxPasoDeTiempo: boolean; EmaxPasoDeTiempo: NReal);

    function Rec: TCosa_RecLnk; override;
    procedure BeforeRead(version, id_hilo: integer); override;
    procedure AfterRead(f:TArchiTexto); override;

    procedure generarLineaResumen(var archi: TextFile); override;
    function infoAd_: string; override;
    procedure Free; override;
  end;

  // TGTer_OnOffPorPaso es una potencia minima y maxima constante en todos los postes
  // y un costo variable cv [USD/MWh] tambien igual en todos los postes
  // y un costo variable cv_min "hasta el minimo" igual al costo de produccion en USD/MWh cuando
  // la central esta generado en el minimo.

  { TGTer_ArranqueParada_noopt }

  TGTer_ArranqueParada_noopt = class(TGTer)

  public

    (**************************************************************************)
    (*               A T R I B U T O S   P E R S I S T E N T E S              *)
    (**************************************************************************)

    encendidoAlInicio: boolean;

    (**************************************************************************)

    pa: TFichaGTer_ArranqueParada_noopt;

    considerarEstadoParaOPT: boolean;
    NMaquinasDespachadasAntesDelPaso: integer;
    NMaquinasDespachadas: integer;
    // cantidad de máquinas despachadas ( variable de Acople )
    CostoDelPaso: NReal; // energía + arranque_parada

    c0: NReal;//Costo en USD/h por máquina por estar operando en el mínimo técnico
    cv: NReal; // USD/MWh usado para la optimización del paso.
    PMin: NReal; //MW, mínimo
    PxMax: NReal; // MW, cota sup de P-Pmin

    // resultado de los sorteos de disponibilidad y del mantenimiento programado
    NMaquinasDisponibles: integer;
    PMaxDisponible: NReal;

    //Cuanta potencia despacho en cada hora el generador en promedio
    potMedia_despachada: NReal;

    constructor Create(capa: integer; nombre: string; nacimiento, muerte: TFecha;
      lpd: TFichasLPD; nodo: TNodo; flg_CalcularGradienteDeInversion: boolean;
      encendidoAlInicio: boolean; TonCO2xMWh: NReal;
      LowCostMustRun, CleanDevelopmentMechanism: boolean); reintroduce;

    function Rec: TCosa_RecLnk; override;
    procedure BeforeRead(version, id_hilo: integer); override;
    procedure AfterRead(f:TArchiTexto); override;

    procedure PrepararMemoria(Catalogo: TCatalogoReferencias; globs: TGlobs); override;
    procedure RegistrarParametrosDinamicos(Catalogo: TCatalogoReferencias); override;
    function PotenciaFirme: NReal; override;
    function InfoAd_: string; override;
    class function DescClase: string; override;
    procedure PosicionarseEnEstrellita; override;
    procedure ActualizarEstadoGlobal(flg_Xs: boolean); override;

    //      procedure Sim_Inicio; override;
    procedure Sim_Cronica_Inicio; override;
    function get_pa_FD(kTipoUnidad: integer): NReal; override;
    function get_pa_TMR(kTipoUnidad: integer): NReal; override;
    procedure SorteosDelPaso(sortear: boolean); override;
    procedure PrepararPaso_ps; override;
    procedure Sim_Paso_Fin; override;
    //      procedure Sim_Cronica_Fin; override;
    //      procedure Sim_Fin; override;

    procedure opt_nvers(var ivar, ivae, ires: integer); override;
    procedure opt_cargue(s: TSimplex); override;
    procedure opt_fijarRestriccionesDeCaja(s: TSimplex); override;
    procedure opt_leerSolucion(s: TSimplex); override;

    function getNombreVar(ivar: integer; var nombre: string): boolean; override;
    function getNombreRes(ires: integer; var nombre: string): boolean; override;
    procedure Free; override;
    procedure PubliVars; override;
    procedure dump_Variables(var f: TextFile; charIndentacion: char); override;
    class function TipoFichaLPD: TClaseDeFichaLPD; override;

  end;

procedure AlInicio;
procedure AlFinal;

implementation

uses uActores;

//-------------------------------------
// Métodos de TFichaGTer_ArranqueParada_noopt
//=====================================

constructor TFichaGTer_ArranqueParada_noopt.Create(capa: integer;
  fecha: TFecha; periodicidad: TPeriodicidad; PMin, PMax: NReal;
  cv_min, cv: NReal; indicePreciosPorCombustible: TFuenteAleatoria;
  bornePreciosPorCombustible: string; disp: NReal; tRepHoras: NReal;
  costo_arranque, costo_parada: Nreal; HayRestriccionEmaxPasoDeTiempo: boolean;
  EmaxPasoDeTiempo: NReal);
begin
  inherited Create(capa, fecha, periodicidad);
  self.PMin := PMin;
  self.PMax := PMax;
  self.cv_min := cv_min;
  self.cv := cv;
  self.disp := disp;
  self.tRepHoras := tRepHoras;
  self.costo_arranque := costo_arranque;
  self.costo_parada := costo_parada;
  self.HayRestriccionEmaxPasoDeTiempo := HayRestriccionEmaxPasoDeTiempo;
  self.EmaxPasoDeTiempo := EmaxPasoDeTiempo;
  self.indicePreciosPorCombustible := indicePreciosPorCombustible;
  self.bornePreciosPorCombustible := bornePreciosPorCombustible;
end;

function TFichaGTer_ArranqueParada_noopt.Rec: TCosa_RecLnk;
begin
  Result := inherited Rec;
  Result.addCampoDef('PMin', PMin, 0, 6);
  Result.addCampoDef('PMax', PMax, 0, 6);
  Result.addCampoDef('cv_min', cv_min, 0, 6);
  Result.addCampoDef('cv', cv, 0, 6);
  Result.addCampoDef_ref('indicePreciosPorCombustible',
    TCosa(indicePreciosPorCombustible), Self, 0, 6);
  Result.addCampoDef('disp', disp, 0, 6);
  Result.addCampoDef('costo_arranque', costo_arranque, 0, 6);
  Result.addCampoDef('costo_parada', costo_parada, 0, 6);
  Result.addCampoDef('HayRestriccionEmaxPasoDeTiempo',
    HayRestriccionEmaxPasoDeTiempo, 0, 6);
  Result.addCampoDef('EmaxPasoDeTiempo', EmaxPasoDeTiempo, 0, 6);
  Result.addCampoDef('PMin', PMin, 6, 14);
  Result.addCampoDef('PMax', PMax, 6, 14);
  Result.addCampoDef('cv_min', cv_min, 6, 14);
  Result.addCampoDef('cv', cv, 6, 14);
  Result.addCampoDef_ref('indicePreciosPorCombustible',
    TCosa(indicePreciosPorCombustible), Self, 6, 14);
  Result.addCampoDef('bornePreciosPorCombustible', bornePreciosPorCombustible, 6, 14);
  Result.addCampoDef('disp', disp, 6, 14);
  Result.addCampoDef('costo_arranque', costo_arranque, 6, 14);
  Result.addCampoDef('costo_parada', costo_parada, 6, 14);
  Result.addCampoDef('HayRestriccionEmaxPasoDeTiempo',
    HayRestriccionEmaxPasoDeTiempo, 6, 14);
  Result.addCampoDef('EmaxPasoDeTiempo', EmaxPasoDeTiempo, 6, 14);
  Result.addCampoDef('PMin', PMin, 14);
  Result.addCampoDef('PMax', PMax, 14);
  Result.addCampoDef('cv_min', cv_min, 14);
  Result.addCampoDef('cv', cv, 14);
  Result.addCampoDef_ref('indicePreciosPorCombustible',
    TCosa(indicePreciosPorCombustible), Self, 14);
  Result.addCampoDef('bornePreciosPorCombustible', bornePreciosPorCombustible, 14);
  Result.addCampoDef('disp', disp, 14);
  Result.addCampoDef('tRepHoras', tRepHoras, 14);
  Result.addCampoDef('costo_arranque', costo_arranque, 14);
  Result.addCampoDef('costo_parada', costo_parada, 14);
  Result.addCampoDef('HayRestriccionEmaxPasoDeTiempo',
    HayRestriccionEmaxPasoDeTiempo, 14);
  Result.addCampoDef('EmaxPasoDeTiempo', EmaxPasoDeTiempo, 14);
end;

procedure TFichaGTer_ArranqueParada_noopt.BeforeRead(version, id_hilo: integer);
begin
  inherited BeforeRead(version, id_hilo);
end;

procedure TFichaGTer_ArranqueParada_noopt.AfterRead(f:TArchiTexto);
begin
  inherited AfterRead(f);
  if f.Version < 6 then
  begin
    self.indicePreciosPorCombustible := nil;
    self.bornePreciosPorCombustible := '';
    self.tRepHoras := 15 * 24;
  end
  else if f.Version < 14 then
    self.tRepHoras := 15 * 24;
end;


procedure TFichaGTer_ArranqueParada_noopt.generarLineaResumen(var archi: TextFile);
begin
  Write(archi, FloatToStrF(PMin, formatoReales, 8, 1), #9,
    //PMín
    FloatToStrF(PMax, formatoReales, 8, 1), #9,
    //PMáx
    FloatToStrF(cv_min, formatoReales, 8, 2), #9,
    //CV_Mín
    FloatToStrF(((cv_min * PMin + cv * (PMax - PMin)) / PMax), formatoReales, 8, 2),
    #9,    //CV_Medio
    FloatToStrF(cv, formatoReales, 8, 2), #9,
    //CV_Incremental
    FloatToStrF(disp, formatoReales, 8, 2), #9,
    //FDisp
    FloatToStrF(costo_arranque, formatoReales, 8, 2), #9,
    //Costo Arranque
    FloatToStrF(costo_parada, formatoReales, 8, 2), #9,
    //Costo Parada
    '-', #9,                                                  //mínNPasosOn
    '-', #9,                                                  //mínNPasosOff
    '-', #9,
    //desiciónPasosOnPorCiclo
    '-', #9,
    //desiciónPasosOffPorCiclo
    '-', #9,
    //costoPorCicloOn
    '-', #9);
  //costoPorCicloOff
end;

function TFichaGTer_ArranqueParada_noopt.infoAd_: string;
begin
  Result := 'PMín= ' + FloatToStrF(PMin, ffGeneral, 10, 1) + ' MW, ' +
    'PMáx= ' + FloatToStrF(PMax, ffGeneral, 10, 1) + ' MW, ' +
    'cv_min= ' + FloatToStrF(cv_min, ffGeneral, 10, 1) + ' USD/MWh, ' +
    'cv= ' + FloatToStrF(cv, ffGeneral, 10, 1) + ' USD/MWh, ' + 'fDisp= ' +
    FloatToStrF(disp, ffGeneral, 10, 1) + ' p.u., ' + 'tRep= ' +
    FloatToStrF(tRepHoras, ffGeneral, 10, 1) + 'h, ' + 'CostoArranque= ' +
    FloatToStrF(costo_arranque, ffGeneral, 10, 1) + ' USD, ' +
    'CostoParada= ' + FloatToStrF(costo_parada, ffGeneral, 10, 1) + ' USD';
end;

procedure TFichaGTer_ArranqueParada_noopt.Free;
begin
  inherited Free;
end;


//--------------------------------
// Métodos de TGTer_ArranqueParada
//================================
procedure TGTer_ArranqueParada_noopt.dump_Variables(var f: TextFile;
  charIndentacion: char);
begin
  inherited dump_Variables(f, charIndentacion);
  writeln(f, charIndentacion, 'c0[USD/MWh]= ', FloatToStrF(c0, ffFixed, 10, 3));
  writeln(f, charIndentacion, 'cv[USD/MWh]= ', FloatToStrF(cv, ffFixed, 10, 3));
  writeln(f, charIndentacion, 'costo_arranque[USD]= ',
    FloatToStrF(pa.costo_arranque, ffFixed, 10, 3));
  writeln(f, charIndentacion, 'costo_parada[USD]= ',
    FloatToStrF(pa.costo_parada, ffFixed, 10, 3));

  writeln(f, charIndentacion, 'PMin[MW]= ', FloatToStrF(PMin, ffFixed, 10, 3));
  writeln(f, charIndentacion, 'PMAx[MW]= ', FloatToStrF(pa.PMax, ffFixed, 10, 3));
  writeln(f, charIndentacion, 'PxMax[MW]= ', FloatToStrF(PxMax, ffFixed, 10, 3));
  writeln(f, charIndentacion, 'PMaxDisponible[MW]= ',
    FloatToStrF(PMaxDisponible, ffFixed, 10, 3));

  writeln(f, charIndentacion, 'NMaquinasDisponibles= ', NMaquinasDisponibles);
  writeln(f, charIndentacion, 'NMaquinasDespachadas= ', NMaquinasDespachadas);
  writeln(f, charIndentacion, 'NMaquinasDespachadasAntesDelPaso= ',
    NMaquinasDespachadasAntesDelPaso);

  writeln(f, charIndentacion, 'HayRestrEMaxPasoDeTiempo= ',
    pa.hayRestriccionEmaxPasoDeTiempo);
  writeln(f, charIndentacion, 'EMaxPasoDeTiempo[MW/h]= ',
    FloatToStrF(pa.EmaxPasoDeTiempo, ffFixed, 10, 3));

  writeln(f);
end;

class function TGTer_ArranqueParada_noopt.TipoFichaLPD: TClaseDeFichaLPD;
begin
  Result := TFichaGTer_ArranqueParada_noopt;
end;


constructor TGTer_ArranqueParada_noopt.Create(capa: integer; nombre: string;
  nacimiento, muerte: TFecha; lpd: TFichasLPD; nodo: TNodo;
  flg_CalcularGradienteDeInversion: boolean; encendidoAlInicio: boolean;
  TonCO2xMWh: NReal; LowCostMustRun, CleanDevelopmentMechanism: boolean);
begin
  inherited Create(capa, nombre, nacimiento, muerte, TActor.CreateDefaultLPDUnidades_(1),
    lpd, nodo,
    flg_CalcularGradienteDeInversion,
    TonCO2xMWh, LowCostMustRun, CleanDevelopmentMechanism, nil, '');
  self.encendidoAlInicio := encendidoAlInicio;
end;

function TGTer_ArranqueParada_noopt.Rec: TCosa_RecLnk;
begin
  Result := inherited Rec;
  Result.addCampoDef('encendidoAlInicio', encendidoAlInicio);
end;

procedure TGTer_ArranqueParada_noopt.BeforeRead(version, id_hilo: integer);
begin
  inherited BeforeRead(version, id_hilo);
end;

procedure TGTer_ArranqueParada_noopt.AfterRead(f:TArchiTexto);
begin
  inherited AfterRead(f);
  pa := nil;
  nodo := nil;
end;

procedure TGTer_ArranqueParada_noopt.PrepararMemoria(Catalogo: TCatalogoReferencias;
  globs: TGlobs);
begin
  inherited prepararMemoria(Catalogo, globs);
  NMaquinasDespachadas := 0;
end;

procedure TGTer_ArranqueParada_noopt.RegistrarParametrosDinamicos(
  Catalogo: TCatalogoReferencias);
var
  i: integer;
  ficha: TFichaGTer_ArranqueParada_noopt;
begin
  inherited registrarParametrosDinamicos(Catalogo);
  lpd.expandirFichas(Catalogo, globs);
  lpd.RegistrarFichasAActualizar(Self, globs.ActualizadorLPD, @pA, nil);
  for i := 0 to lpd.Count - 1 do
  begin
    ficha := TFichaGTer_ArranqueParada_noopt(lpd[i]);
    if ficha.indicePreciosPorCombustible <> nil then
      ficha.nroBornePreciosPorCombustible :=
        ficha.indicePreciosPorCombustible.IdBorne(ficha.bornePreciosPorCombustible);
  end;
end;

function TGTer_ArranqueParada_noopt.PotenciaFirme: NReal;
begin
  Result := (paUnidades.nUnidades_Operativas[0]) * pa.PMax * pa.disp;
end;

function TGTer_ArranqueParada_noopt.InfoAd_: string;
begin
  Result := '';
end;

class function TGTer_ArranqueParada_noopt.DescClase: string;
begin
  Result := rsGeneradorTermicoConCostoArranqueParada;
end;

procedure TGTer_ArranqueParada_noopt.PosicionarseEnEstrellita;
begin
  Self.NMaquinasDespachadas := globs.CF.xd[ixd];
end;

procedure TGTer_ArranqueParada_noopt.ActualizarEstadoGlobal(flg_Xs: boolean);
begin
  globs.CF.xd[ixd] := Self.NMaquinasDespachadas;
end;

procedure TGTer_ArranqueParada_noopt.Sim_Cronica_Inicio;
begin
  inherited Sim_Cronica_Inicio;
  if encendidoAlInicio then
    NMaquinasDespachadas := 1
  else
    NMaquinasDespachadas := 0;
end;


function TGTer_ArranqueParada_noopt.get_pa_FD(kTipoUnidad: integer): NReal;
begin
  Result := pa.disp;
end;

function TGTer_ArranqueParada_noopt.get_pa_TMR(kTipoUnidad: integer): NReal;
begin
  Result := pa.tRepHoras;
end;



procedure TGTer_ArranqueParada_noopt.SorteosDelPaso(sortear: boolean);
begin
  if globs.ObligarDisponibilidad_1_ then
  begin
    NMaquinasDisponibles := paUnidades.nUnidades_Operativas[0];
    PMaxDisponible := pa.PMax * NMaquinasDisponibles;
  end
  else if sortear then
  begin
    ActualizarProbabilidadesReparacionYRotura_(pa.disp, pa.tRepHoras);
    NMaquinasDisponibles := Sorteos_RepRotUnidades;
    PMaxDisponible := pa.PMax * NMaquinasDisponibles;
  end
  else
  begin
    NMaquinasDisponibles := paUnidades.nUnidades_Operativas[0];
    PMaxDisponible := pa.PMax * pa.disp * NMaquinasDisponibles;
  end;
end;

procedure TGTer_ArranqueParada_noopt.PrepararPaso_ps;
var
  indice: NReal;
begin
  PMin := pa.PMIn;
  PxMax := pa.PMax - pa.PMin;
  NMaquinasDespachadasAntesDelPaso := NMaquinasDespachadas;
  if pa.indicePreciosPorCombustible <> nil then
  begin
    indice := pa.indicePreciosPorCombustible.bornera[pa.nroBornePreciosPorCombustible];
    c0 := pa.PMin * pa.cv_min * indice;
    cv := pa.cv * indice;
  end
  else
  begin
    c0 := pa.PMin * pa.cv_min;
    cv := pa.cv;
  end;
end;

procedure TGTer_ArranqueParada_noopt.Sim_Paso_Fin;
var
  iposte: integer;
begin
  for iposte := 0 to high(P) do
    potMedia_despachada := potMedia_despachada + P[iposte] * globs.durpos[iposte];
  potMedia_despachada := potMedia_despachada * globs.invHorasDelPaso;
end;


procedure TGTer_ArranqueParada_noopt.opt_nvers(var ivar, ivae, ires: integer);
begin
  if NMaquinasDisponibles = 0 then
    exit;
  Self.ivar := ivar;
  ivar := ivar + globs.NPostes + 1; //Potencias y A
  Self.ivae := ivae;
  Self.ires := ires;
  ires := ires + globs.NPostes;     //Restricciones de minimos técnicos
  if pA.HayRestriccionEmaxPasoDeTiempo then
    Inc(ires);
end;


procedure TGTer_ArranqueParada_noopt.opt_cargue(s: TSimplex);
var
  inodores: integer;
  iposte: integer;
  jres: integer;
begin
  if NMaquinasDisponibles = 0 then
    exit; // si no hay máquinas no juego

  inodores := nodo.ires;
  // CON MINIMOS TECNICOS
  // aporte a las restricciones de nodo

  for iposte := 0 to globs.NPostes - 1 do
  begin
    s.pon_e(inodores + iposte, ivar + iposte, 1); // coeficiente de la B[iposte]
    s.pon_e(inodores + iposte, ivar + globs.NPostes, Pmin); // coef A
  end;

  jres := ires;
  // restricciones adicionales impuestas por la variable de Acople
  for iposte := 0 to globs.NPostes - 1 do
  begin
    s.pon_e(jres, ivar + iposte, -1); // coef B[iposte]
    s.pon_e(jres, ivar + globs.NPostes, PxMax);
    Inc(jres);
  end;
  if pA.HayRestriccionEmaxPasoDeTiempo then
  begin
    for iposte := 0 to globs.NPostes - 1 do
      s.pon_e(jres, ivar + iposte, -globs.durpos[iposte]);
    s.pon_e(jres, s.nc, pA.EmaxPasoDeTiempo - pa.PMin);
  end;

  // aportes a la función de utilidad
  for iposte := 0 to globs.NPostes - 1 do
    s.pon_e(s.nf, ivar + iposte, -cv * globs.DurPos[iposte]);
  s.pon_e(s.nf, ivar + globs.NPostes, -c0 * globs.HorasDelPaso);

  // controlamos el uso correcto del modelo.
  if NMaquinasDisponibles > 1 then
    raise Exception.Create(
      'Por ahora TGer_ArranqueParada soporta solamente centrales de una sóla máquina');

  // agregar costos de Arranque o Parda según corresponda
  if NMaquinasDespachadas = 0 then // estaba apagada
  begin
    s.acum_e(s.nf, ivar + globs.NPostes, -pa.costo_Arranque);
  end
  else // estaba prendida
  begin
    s.acum_e(s.nf, ivar + globs.NPostes, pa.costo_Parada);
    s.acum_e(s.nf, s.nc, -pa.costo_Parada);
  end;
end;

procedure TGTer_ArranqueParada_noopt.opt_fijarRestriccionesDeCaja(s: TSimplex);
var
  acoples: TListaAcoplesVEntera;
  iposte: integer;
begin
  if NMaquinasDisponibles = 0 then
    exit;
  (*
    // Restricciones de caja de las B (Esto no es necesario pues las restricciones adicionales
    // obligan que estas se cumplan. Por eso le pongo el 1.1 para dejarle la caja un poco floja
    for iposte:= 0 to globs.NPostes-1 do
      s.cota_sup_set( ivar+iposte, PxMax*NMaquinasDisponibles*1.1 );
  *)
  // Restricciones de caja de las A y las declaramos enteras
  setLength(acoples, globs.NPostes);
  for iposte := 0 to globs.NPostes - 1 do
  begin
    acoples[iposte].ivar := ivar + iposte;
    acoples[iposte].ires := ires + iposte;
  end;
  // Restricciones de caja de las A y las declaramos enteras
  TMIPSimplex(s).set_EnteraConAcoples(
    ivae, ivar + globs.NPostes, NMaquinasDisponibles, acoples);
end;

procedure TGTer_ArranqueParada_noopt.opt_leerSolucion(s: TSimplex);
var
  iposte: integer;
  EnergiaPorEncimaDelMinimo: NReal;
  m: NReal;
begin

  CostoDirectoDelPaso := 0;
  if NMaquinasDisponibles = 0 then
  begin
    vclear(P);
    //vclear( cv_Spot );
    for iposte := 0 to globs.NPostes - 1 do
    begin
      Lambda_P[iPoste] := 0;
      cv_Spot[iPoste] := get_cv_falla;
      // para que en el ordenamiento quede último, con costo variable igual a la falla;
    end;
    NMaquinasDespachadas := 0;
  end
  else
  begin
    NMaquinasDespachadas := trunc(s.xval(ivar + globs.NPostes) + 0.2);
    if NMaquinasDespachadas = 0 then
    begin
      vclear(P);
      //vclear( cv_Spot );
      for iposte := 0 to globs.NPostes - 1 do
      begin
        Lambda_P[iPoste] := 0;
        cv_Spot[iPoste] := get_cv_falla;
        // para que en el ordenamiento quede último, con costo variable igual a la falla;
      end;
      exit;
    end;

    EnergiaPorEncimaDelMinimo := 0;
    for iposte := 0 to globs.NPostes - 1 do
    begin
      m := s.xval(ivar + iposte);
      P[iposte] := m + NMaquinasDespachadas * Pmin;
      Lambda_P[iPoste] := s.xmult(ires + iposte) / globs.DurPos[iposte];
      cv_Spot[iPoste] := Nodo.cmarg[iposte] - Lambda_P[iPoste];
      EnergiaPorEncimaDelMinimo := EnergiaPorEncimaDelMinimo + m * globs.DurPos[iposte];
    end;

    CostoDirectoDelPaso := EnergiaPorEncimaDelMinimo * cv +
      NMaquinasDespachadas * c0 * globs.HorasDelPaso;

    if NMaquinasDespachadasAntesDelPaso < NMaquinasDespachadas then
      CostoDirectoDelPaso := CostoDirectoDelPaso +
        (NMaquinasDespachadas - NMaquinasDespachadasAntesDelPaso) * pa.costo_arranque
    else if NMaquinasDespachadasAntesDelPaso > NMaquinasDespachadas then
      CostoDirectoDelPaso := CostoDirectoDelPaso +
        (NMaquinasDespachadasAntesDelPaso - NMaquinasDespachadas) * pa.costo_parada;
  end;
end;


function TGTer_ArranqueParada_noopt.getNombreVar(ivar: integer;
  var nombre: string): boolean;
begin
  if NMaquinasDisponibles = 0 then
    Result := False
  else
  if (ivar >= self.ivar) and (ivar < self.ivar + globs.NPostes) then
  begin
    nombre := self.Nombre + '_P[MW]' + IntToStr(ivar - self.ivar + 1);
    Result := True;
  end
  else if (ivar = self.ivar + globs.NPostes) then
  begin
    nombre := self.Nombre + '_A[MW]';
    Result := True;
  end
  else
    Result := False;
end;

function TGTer_ArranqueParada_noopt.getNombreRes(ires: integer;
  var nombre: string): boolean;
begin
  if NMaquinasDisponibles = 0 then
    Result := False
  else
  if (ires >= self.ires) and (ires < self.ires + globs.NPostes) then
  begin
    nombre := self.nombre + '_res-A' + IntToStr(ires - self.ires + 1);
    Result := True;
  end
  else if pa.HayRestriccionEmaxPasoDeTiempo and (ires = self.ires + globs.NPostes) then
  begin
    nombre := self.nombre + '_res-EMax';
    Result := True;
  end
  else
    Result := False;
end;

procedure TGTer_ArranqueParada_noopt.PubliVars;
begin
  inherited PubliVars;

  PublicarVariableNR('Costo', '[USD]', 6, 1, costoDirectoDelPaso, True);
  PublicarVariableNI('NMaqsDespachadas', '-', NMaquinasDespachadas, True);

  PublicarVariableNR('c0', '[USD/h]', 6, 1, c0, False);
  PublicarVariableNI('NMaqsDisponibles', '-', NMaquinasDisponibles, False);
  PublicarVariableNR('PMaxDisponible', '[MW]', 6, 1, PMaxDisponible, False);
  PublicarVariableNR('PMediaDespachada', '[MW]', 6, 1, potMedia_despachada, False);
end;

procedure TGTer_ArranqueParada_noopt.Free;
begin
  inherited Free;
end;

procedure AlInicio;
begin
  registrarClaseDeCosa(TGTer_ArranqueParada_noopt.ClassName,
    TGTer_ArranqueParada_noopt);
  registrarClaseDeCosa(TFichaGTer_ArranqueParada_noopt.ClassName,
    TFichaGTer_ArranqueParada_noopt);
end;

procedure AlFinal;
begin

end;

end.
