unit uHidroConEmbalseValorizado;
(*

El embalse valorizado nos permite definir los valores del agua
sin tener que agregar una variable de estado.
Puede usarse para las centrales a pelo de agua o centrales prcticamente
de pasada.

La idea es que en este caso el embalse no se considera una variable de
estado del sistema y no interviene como tal en el problema de optimizacin.

*)

interface
uses
	Math, xMatDefs, uSimplex, ucosa, uGlobs, uFechas, uNodos,
	sysutils,
	uRandDispos,
	uFichasLPD,
	uGeneradores,
	uconstantesSimSEE, uauxiliares,
  uFuentesAleatorias,
  uCosaConNombre;

const
	ro_Agua= 1000; // kg/m3 densidad del agua
	g_Gravitacion= 9.8; // N/kg constante gravitatoria

type
	 THidroConEmbalseValorizado= class; // forward declaration

(* Para describir el embalse, damos el volumen mximo turbinalbe Vmax
que corresponde al volumen (en m3 ) que almacena el embalse entre las
cotas hmin y hmax.
Y el volumen almacenado entre hmin y hmed siendo hmed = (hmax+hmin)/2

Las cotas estn medidas desde algn nivel (por ejemplo el del mar).
Para poder calcular el salto se suministra tambin la cota de aguas abajo
o de descarga de la turbina referida al mismo nivel. (hDescarga [m])
El nivel de descarga puede ser especificado dando su valor con la variable
hDescarga o mediante la referencia a otra central en cuyo caso se considera
que el nivel de descarga es el del lago de la otra central ms el valor
hDescarga. En este caso la variable hDescarga se utiliza para tener en
cuenta cambios en la referencia de niven entre una central y otra.
Si todas las cotas estn referidas al nivel del mar no hay cambios de
referencias y hDescarga se pondr en cero.

Suponemos que el rea del embalse crece linealmente con la cota
A(z) = a + b * z
V(z) = integral( A(z) )= a*z + 1/2 * b z^2

con z= (h-hmin)/(hmax-hmin)

Vmax= V(1) = a+ 0.5 b
Vhmed= V(0.5) = 0.5 a + 0.5^3 b

=> Vmax - 2*Vhmed= 0.5 b - 0.5^2 b =  0.5^2 b
=>  b = 4*(Vmax-2*Vhmed)
.:  a = Vmax - 0.5 b



V= ( 1/2*b/dhmax^2 * dh + a/dhmax ) dh =


Prdida de Salto por Caudal Erogado.
====================================
El caudal erogado QErogado [m3/s] es calculado al final de cada paso
de simulacin como la suma del caudal vertido ms el caudal turbinado.
Dicho caudal puede afectar el nivel de aguas abajo reducciendo el salto
efectivo en la central. Para tener en cuenta este efecto se usan los
parmetros: caQE y cbQE calculados para modelar la prdida de salto
de acuerdo a la siguiente ecuacin:

dh(QE) = caQE* QE + cbQE* QE^2
*)
	TFichaHidroConEmbalseValorizado = class (TFichaLPD)
		public
			hmax: NReal; //[m]= 10; // cota maxima
			hmin: NReal; //[m]= 2; // cota minima
			Vmax: NReal; //[m3] volumen mximo almacenable
			Vhmed: NReal; //[m3] volumen con la cota media

      centralesLagoArriba : TListaCentralesAguasArriba{of IGeneradorHidraulico};
			hDescarga: NReal; //[m] cota de la descarga para clculo del salto
			central_lagoDescarga: TGeneradorHidraulico;
			// Coeficientes de afectacin del salto por caudal erogado
			// dh(QE) = caQE* QE + cbQE* QE^2
			caQE: NReal;
			cbQE: NReal;


				 (** DATOS CENTRALES URUGUAY
				 BONETE= Dr. Gabriel Terra
				 =========================
				 hmax= 86.5
				 hmin= 70
				 Vmax= 1.667E+16
				 Vhmed= 8.335E+15
				 hDescarga= 54.5394
				 caQE= 4.689E-04
				 cbQE= 5.077E-07

				 ren= 0.85 ??? ojo ver mejor este nmero
				 Pmax_Gen= 38.8
				 Qmax_Turb= 170.0
				 fDispo= 0.995

				 *)

			ren: NReal; //[pu]= 0.95; // rendimiento complexivo de turbina y generador
			Pmax_Gen: NReal; //[MW]= 240; // POtencia maxima hidraulica
			Qmax_Turb: NReal; //[m3/s]
			fDispo: NReal; //[pu]  factor de disponibilidad fortuito
			ca_filtr, cb_filtr: NReal; //filtracin[m^3/s] = ca_filtr + cb_filtr * h

			HayRestriccionEmaxPasoDeTiempo: boolean; // indica si se aplica la restriccin
			EmaxPasoDeTiempo: NReal; // Energa maxima generable en un paso de tiempo

			HayRestriccionQTmin: boolean; // indica si se aplica la restriccin
			QTmin: NReal; // caudal mnimo para asegurar navegabilidad

			ValorDelAgua: TDAOfNReal;

			Constructor Create(  fecha : TFecha; periodicidad : TPeriodicidad ;
															hmin_,	hmax_, Vmax_,	V_hmed, // volumen con la cota media
															hDescarga: NReal;
															central_lagoDescarga: THidroConEmbalseValorizado;
                              centralesLagoArriba : TListaCentralesAguasArriba;
															caQE,  cbQE,  ren_, Pmax_Gen_,	Qmax_Turb_: NReal;
															fDispo_: NReal;
															ca_filtr, cb_filtr: NReal;
															ValorDelAgua: TDAOfNReal );
			Constructor Create_ReadFromText(f : TArchiTexto); override;
			procedure WriteToText( f: TArchiTexto ); override;
			procedure Free; override;
			end;


	THidroConEmbalseValorizado = class( TGeneradorHidraulico )
		public
			pa : TFichaHidroConEmbalseValorizado;
			lpd : TFichasLPD;
      fuente : TFuenteAleatoria;
      numeroBorne : Integer;
      nombreBorne : String;

			//Variables de Estado
			NDisc: integer; //= 10; // discretizacion del estado
			hini : NReal; // cota inicial

//----------- variables auxiliares
			dx: NReal; //= VMax / ( NDisc -1 ); // escalon de discretizacion
//			frv: NReal; // factor de reduccion del volumen en el paso de tiempo
			ro_g_ren: NReal; // ro_Agua * g_gravitacion * ren /(1E6*3600)
			dhmax: NReal; // hmax - hmin
			VTmin: NReal; // volumen turbinado mnimo en el paso para cumplir con QTmin

// variables calculadas por paso de tiempo durante la simulacin
			P: TDAOfNReal; // potencias despachadas
			Vertimiento: NReal; // Volumen vertido en el paso de tiempo
			h_actual: NReal; // cota actual
			Salto: NReal; // salto actual efectivo

			EnergiaGenerada: NReal; // [MWh] Total de energa generada en el paso
			QErogado: NReal; // Caudal erogado en m3/s

			VolumenActual: NReal; // volumen actual En el embalse valorizado no la consideramos una
								// variable de estado del sistema.

			xsVol: NReal; // volumen resultado del paso. Asignado al estado en FinPaso
			PmaxParaQmax: NReal; // Pontecia mxima por mquina considerando Qmax y la cota
			NMaquinasDisponibles: integer;
			Pmax_Central: NReal; // [MW] Potencia Mxima disponible en la central
			Pmax_Gen: NReal; // [MW] Potencia mxima generable por turbina-generador
			Qmax: NReal; // [m3/s] Caudal mximo turbinable por turbina-generador
			PuedoGenerar: boolean; // indica si la Central est en condiciones de ofrecer potencia o no
			cv_agua_USD_MWh, cv_agua_USD_m3: NReal; // valor del Agua expresado en USD/MWh

			aporte: NReal; // volumen de aporte en el paso de tiempo

// variables auxiliares para las funciones cota-volumen y su inversa
			EsConico: boolean;
			 embalse_cb, embalse_ca, embalse_c1, embalse_c2,
			 embalse_c3, embalse_c4, embalse_c5: NReal;

			constructor Create(
				nombre: string;
				nacimiento, muerte: TFecha;
        lpdUnidades: TFichasLPD;
				nodo: TNodo ; hIni : NReal ; ficha : TFichaHidroConEmbalseValorizado;
        fuente : TFuenteAleatoria ; nombreBorne : String);

			constructor Create_ReadFromText( f: TArchiTexto ); override;
			procedure WriteToText( f: TArchiTexto ); override;
			procedure PrepararMemoria(globs : TGlobs);override;
			procedure RegistrarParametrosDinamicos; override;

			function PotenciaFirme : NReal;	override;

			function InfoAd : String;	override;
			class function DescClase : String; override;

//			procedure Sim_Inicio; override;
      procedure Sim_Cronica_Inicio; override;
			procedure SorteosDelPaso( sortear: boolean ); override;
			procedure PrepararPaso_ps; override;
			procedure Sim_Paso_Fin_; override;

			procedure opt_nvers( var ivar, ivae, ires: integer ); override;

			procedure opt_cargue( s: TSimplex ); override;
			procedure opt_fijarRestriccionesDeCaja( s: TSimplex ); override;
			procedure opt_leerSolucion( s: TSimplex ); override;
(*
			procedure optx_nvrxs( var irx: integer ); override;
			procedure optx_nvdxs( var iex: integer ); override;
			procedure optx_RegistrarVariablesDeEstado(  adminEstados: TAdminEstados ); override;
			*)

			procedure Free; override;

		// geometra del embalse
			function CotaToVolumen( h: NReal ): NReal;
			function VolumenToCota( Vol: NReal ): NReal;

			procedure CambioFichaPD; override;

			(*
			procedure SetVolumen( xVol: NReal );
			function GetVolumen: NReal;
			property V: NReal read GetVolumen write SetVolumen;
				*)
//		function CostoDelPaso: NReal; override;
			procedure PubliVars; override;

// Calcula la cota de la descarga para poder calcular el Salto
			function CotaDescarga: NReal; override;

// Caclula la reduccin del salto causada por el caudal Erogado
			function ReduccionDeCotaPorCaudalErogado( QErogado:NReal ): NReal; override;
	end;

procedure THidroConEmbalseValorizado_cambioFichaPD(Actor : TCosa);

procedure AlInicio;
procedure AlFinal;

implementation

constructor THidroConEmbalseValorizado.Create(
							nombre: string;
							nacimiento, muerte: TFecha;
              lpdUnidades: TFichasLPD;
							nodo: TNodo; hIni: NReal;
              ficha : TFichaHidroConEmbalseValorizado;
              fuente : TFuenteAleatoria ; nombreBorne : String);

begin
	inherited Create( nombre, nacimiento, muerte, lpdUnidades, nodo);
	Self.lpd := TFichasLPD.Create(self, TFichaHidroConEmbalseValorizado);
	self.hini := hIni;
  self.fuente := fuente;
  self.nombreBorne := nombreBorne;
	if ficha <> NIL then
		lpd.Add(ficha);
end;

constructor THidroConEmbalseValorizado.Create_ReadFromText( f: TArchiTexto );
begin
	inherited Create_ReadFromText( f );
  f.IniciarLecturaRetrasada;
	f.rd('hIni', hIni);
	f.rd('lpd', TCosa(lpd));
  f.rdReferencia('fuente', TCosa(fuente), Self);
  f.rd('nombreBorne', nombreBorne);
  f.EjecutarLectura;
	pa := NIL;
	nodo := NIL;
  lpd.Propietario:= self;  
end;

procedure THidroConEmbalseValorizado.WriteToText( f: TArchiTexto );
begin
	inherited WriteToText( f );
	f.wr('hIni', hini, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES);
	f.wr('lpd', lpd);
  f.wrReferencia('fuente', fuente);
  f.wr('nombreBorne', nombreBorne);
end;

procedure THidroConEmbalseValorizado.prepararMemoria(globs : TGlobs);
begin
	 inherited prepararMemoria(globs );
	 setlength( P, globs.NPostes );
end;

procedure THidroConEmbalseValorizado.registrarParametrosDinamicos;
begin
	inherited registrarParametrosDinamicos;
 	lpd.expandirFichas(globs);
	lpd.RegistrarFichasAActualizar(Self, globs.ActualizadorLPD, @pA, nil, THidroConEmbalseValorizado_cambioFichaPD );
end;

function THidroConEmbalseValorizado.infoAd : String;
begin
	result := '';
end;

function THidroConEmbalseValorizado.potenciaFirme : NReal;
begin
	result := (paUnidades.nUnidades)* pa.Pmax_Gen * pa.ren;
end;

class function THidroConEmbalseValorizado.DescClase : String;
begin
	result := 'Generador Hidrulico Con Embalse Valorizado';
end;

procedure THidroConEmbalseValorizado.SorteosDelPaso( sortear: boolean );
begin
	if ( sortear ) then
  begin
	  NMaquinasDisponibles:= Sorteos_RepRotUnidades;
		Pmax_Gen:= pA.Pmax_gen;
		Qmax:= pA.Qmax_Turb;
// POR AHORA PONEMOS EL APORTE COMO UN SORTEO
		Aporte:= paUnidades.nUnidades* pA.Qmax_turb* globs.SegundosDelPaso*( 2* globs.sorteadorUniforme.rnd );
	end
	else
	begin
		NMaquinasDisponibles:= paUnidades.nUnidades;
		Pmax_Gen:= pA.Pmax_gen*pA.fdispo;
		Qmax:= pA.Qmax_Turb* pA.fdispo;
//Para el aporte medio ponemos el valor esperado del aporte
		Aporte:= paUnidades.nUnidades * pA.Qmax_turb* globs.SegundosDelPaso ;
	end;
end;

function THidroConEmbalseValorizado.ReduccionDeCotaPorCaudalErogado( QErogado: NReal ): NReal;
begin
	 result:= ( pa.cbQE * QErogado + pa.caQE )* QErogado;
end;

function THidroConEmbalseValorizado.CotaDescarga: NReal;
begin
	 if pa.central_lagoDescarga= nil then
			result:= pa.hDescarga
	 else
//			result:= pa.central_LagoDescarga.h_actual;
			result:= pa.central_LagoDescarga.CotaDescarga;
end;



procedure THidroConEmbalseValorizado.prepararPaso_ps;


begin
	h_actual:= VolumenToCota( VolumenActual );
	Salto:= h_actual - CotaDescarga - ReduccionDeCotaPorCaudalErogado( QErogado );

	ce:= Salto* ro_g_ren;

	cv_agua_USD_MWh:= interpolar( pA.valorDelAgua,  pA.hMax, pa.hmin, h_actual );
	cv_agua_USD_m3:= cv_agua_USD_MWh * ce;


	 // factor de reduccin del volumen embalsados por filtracin y
	 // evaporacin en un paso de tiempo.
//	 frv:= Math.Power( pA.frv_ef10, globs.HorasDelPaso/240 );

	 PmaxParaQmax:= Qmax * ce *3600;

	 if NMaquinasDisponibles > 0 then
			Pmax_Central:= NMaquinasDisponibles * min( Pmax_Gen, PMaxParaQmax )
	 else
			PMax_Central:= 0;

	 if PMax_Central > 0.1 then
		PuedoGenerar:= true
	 else
		PuedoGenerar:= false;

	 if pA.HayRestriccionQTmin then
		 VTmin:= PA.QTmin * globs.SegundosDelPaso;
end;

procedure THidroConEmbalseValorizado.opt_nvers( var ivar, ivae, ires: integer );
begin
	self.ivar:= ivar;
	ivar:= ivar+ globs.NPostes + 1; // PGen por poste y el QVertido  total

	self.ires:= ires;
	ires:= ires+2;// 0 <= volumen <= Vmx  ( dos restricciones )
	if pA.HayRestriccionEmaxPasoDeTiempo then inc( ires );
	if pA.HayRestriccionQTmin then inc( ires );
end;

procedure THidroConEmbalseValorizado.opt_cargue( s: TSimplex );
var
	ibaseres: integer;
	iposte: integer;
	jres: integer;
	ivert: integer;
begin
	ivert:= ivar+globs.NPostes;

{$IFDEF SPXCONLOG}
	for iposte:= 0 to globs.NPostes-1 do
		s.set_NombreVar( ivar+iposte, Nombre+'_P[MW]'+IntToStr(iposte+1) );

	s.set_NombreVar( ivar+globs.NPostes, Nombre+'_Vertimiento[m3]' );
	jres:= ires;
	s.set_NombreRest( jres, Nombre+'_VOL>=0 ' );
	inc( jres );
	s.set_NombreRest( jres, Nombre+'_VOL<=Vmax ');
	inc( jres );
	if pA.HayRestriccionEmaxPasoDeTiempo then
	begin
		s.set_NombreRest( jres, Nombre+'_Emax' );
		inc( jres );
	end;

	if pA.HayRestriccionQTmin then
	begin
		s.set_NombreRest( jres, Nombre+'_QTmin' );
	end;
{$ENDIF}

	ibaseres:= nodo.ires;
	// Tenemos que aportar a las restricciones de demanda del nodo al que est
	// conectado el generador

	for iposte:= 0 to globs.NPostes-1 do
		s.pon_e( ibaseres+iposte, ivar+iposte , 1 );

	// Ahora agregamos a la funcin de UTILIDAD (-costo)
	for iposte:= 0 to globs.NPostes-1 do
		s.pon_e( s.nf, ivar+iposte , -cv_agua_USD_MWh* globs.durpos[ iposte ] );
	s.pon_e( s.nf, ivert, -cv_agua_USD_m3 ); //  Vertimiento
	s.acum_e( s.nf, s.nc, +cv_agua_USD_m3*Aporte ); // Aporte

// restriccion de volumen >= 0 = Vmn
	jres:= ires;
	for iposte:= 0 to globs.NPostes-1 do
		s.pon_e( jres, ivar+iposte, -globs.durpos[iposte]/ce );
	s.pon_e( jres, ivert, -1 );  // vertimiento
//	s.pon_e( jres, s.nc,  (VolumenActual * frv + Aporte)  );
  s.pon_e( jres, s.nc,  (VolumenActual - (pa.ca_filtr + pa.cb_filtr * h_actual) + Aporte)  );
  
// restriccin de volumen <= Vmax
	inc( jres );
	for iposte:= 0 to globs.NPostes-1 do
		s.pon_e( jres, ivar+iposte, globs.durpos[iposte]/ce );
	s.pon_e( jres, ivert, 1 );  // vertimiento
//	s.pon_e( jres, s.nc,  (pA.Vmax -(VolumenActual * frv + Aporte))  );
  s.pon_e( jres, s.nc,  (pA.Vmax -(VolumenActual - (pa.ca_filtr + pa.cb_filtr * h_actual) + Aporte))  );

	if pA.HayRestriccionEmaxPasoDeTiempo then
	begin
		inc( jres );
		for iposte:= 0 to globs.NPostes-1 do
			s.pon_e( jres, ivar+iposte, -globs.durpos[iposte] );
		s.pon_e( jres, s.nc,  pA.EmaxPasoDeTiempo );
	end;

	if pA.HayRestriccionQTmin then
	begin
		inc( jres );
		for iposte:= 0 to globs.NPostes-1 do
			s.pon_e( jres, ivar+iposte, globs.durpos[iposte]/ce );
		s.pon_e( jres, ivert, 1 );  // vertimiento
		s.pon_e( jres, s.nc,  -VTmin  );
	end;
end;

procedure THidroConEmbalseValorizado.opt_fijarRestriccionesDeCaja( s: TSimplex );
var
	iposte: integer;
begin
	// Le fijamos como cota mxima PMax a la potencia en todos los postes
	if PuedoGenerar then
	begin
		for iposte:= 0 to globs.NPostes-1 do
				s.cota_sup_set( ivar+iposte, PMax_Central );
	end
	else
	begin
		for iposte:= 0 to globs.NPostes-1 do
			s.FijarVariable( ivar+iposte, 0);
	end;
end;

procedure THidroConEmbalseValorizado.Sim_Cronica_Inicio;
begin
  inherited Sim_Cronica_Inicio;
	QErogado:= 0;
	Vertimiento:= 0;
	EnergiaGenerada:= 0;
	VolumenActual:= CotaToVolumen( hini );
end;

procedure THidroConEmbalseValorizado.opt_leerSolucion( s: TSimplex );
var
	iposte: integer;
begin
	if PuedoGenerar then
	begin
	// recuperamos los valores de Potencia despachada
			EnergiaGenerada:= 0;
		for iposte:= 0 to globs.NPostes-1 do
			begin
			P[iposte]:= s.xval( ivar+iposte );
				 EnergiaGenerada:= EnergiaGenerada + P[iposte] * globs.durpos[iposte];
			end;
		Vertimiento:= s.xval( ivar+globs.NPostes );
	end
	else
	begin
		for iposte:= 0 to globs.NPostes-1 do
			P[iposte]:= 0;
			EnergiaGenerada:= 0;
		Vertimiento:= s.xval( ivar );
	end;
  QErogado:= (EnergiaGenerada / ce + Vertimiento) * globs.invSegundosDelPaso;
	xsVol:= s.yval( ires );
end;

(*
function THidroConEmbalseValorizado.CostoDelPaso: NReal;
var
	costo: NReal;
	iposte: integer;
begin
	costo:= 0;
	for iposte:= 0 to globs.NPostes-1 do
			costo:= costo+P[iposte]*cv_agua_USD_MWh*globs.durpos[iposte];
	result:= costo;
end;
	*)

procedure THidroConEmbalseValorizado.PubliVars;
begin
	inherited PubliVars;
	PublicarVariableVR('P', '[MW]', 6, 1, P, true);
	PublicarVariableNR('Vertimiento', '[Hm^3]', 8, 1, Vertimiento);
	PublicarVariableNR('Cota', '[m]', 6, 1, h_actual);
	PublicarVariableNR('Vol', '[m^3]', 8, 1, xsVol);
	PublicarVariableNI('NMaqsDisponibles', '-', NMaquinasDisponibles);
	PublicarVariableNR('cv_agua', '[USD/MWh]', 6, 1, cv_agua_USD_MWh);
	PublicarVariableNR('QAportes', '[m^3/s]', 8, 1, aporte);
end;

procedure THidroConEmbalseValorizado.Sim_Paso_Fin;
begin
	VolumenActual:= xsVol;
end;

procedure THidroConEmbalseValorizado.free;
begin
	lpd.Free;
	inherited Free;
end;

function THidroConEmbalseValorizado.CotaToVolumen( h: NReal ): NReal;
var
	dh: NReal;
begin
	dh:= h- pA.hmin;
	if EsConico then
		result:= ( embalse_c2 * dh + embalse_c1 ) * dh
	else
		result:= embalse_c1* dh;
end;

function THidroConEmbalseValorizado.VolumenToCota( Vol: NReal ): NReal;
begin
	if EsConico then
		result:= sqrt( embalse_c3 * Vol + embalse_c4 )+ embalse_c5
	else
		result:= Vol / embalse_c1 + pA.hmin;
end;

procedure THidroConEmbalseValorizado.cambioFichaPD;
begin
  ActualizarProbabilidadesReparacionYRotura( pa.fDispo, 48);
	NDisc:= length( pA.ValorDelAgua );
	dx := pA.VMax / ( NDisc -1 ); // escalon de discretizacion
	ro_g_ren :=  pA.ren * ro_Agua * g_gravitacion/1.0E6/3600.0;

  dhmax:= ( pA.hmax - pA.hmin );
	embalse_cb:= 4.0* ( pA.Vmax - 2.0* pA.Vhmed );
	embalse_ca:= pA.Vmax - embalse_cb / 2.0;
	embalse_c1:= embalse_ca / dhmax ;

	if EsCero( embalse_cb ) then
	begin
	  EsConico:= false;
  end
  else
	begin
	  EsConico:= true;
		embalse_c2:= embalse_cb / 2.0 / dhmax;

		embalse_c3:= (2.0 / embalse_cb) * sqr(dhmax);
		embalse_c4:= sqr( embalse_ca / embalse_cb ) * sqr(dhmax);
		embalse_c5:= pA.hmin - embalse_ca / embalse_cb * dhmax;
  end;
end;

{*********************************
*Mtodos de TFichaHidroConEmbalse*
*********************************}

Constructor TFichaHidroConEmbalseValorizado.Create(
	 fecha : TFecha; periodicidad : TPeriodicidad ;
	 hmin_,	hmax_, Vmax_,	V_hmed, // volumen con la cota media
	 hDescarga: NReal;
	 central_lagoDescarga: THidroConEmbalseValorizado;
   centralesLagoArriba : TListaCentralesAguasArriba;
	 caQE,  cbQE,
	 ren_, Pmax_Gen_,	Qmax_Turb_: NReal;
	 fDispo_: NReal; ca_filtr, cb_filtr: NReal;
	 ValorDelAgua: TDAOfNReal);
begin
	inherited Create(fecha, periodicidad);
	self.hmin := hmin_;
	self.hmax := hmax_;
	self.Vmax := Vmax_;
	self.Vhmed := V_hmed;
	self.hDescarga:= hDescarga;
	self.central_lagoDescarga:= central_lagoDescarga;
	self.caQE:= caQE;
	self.cbQE:= cbQE;
	self.ren := ren_;
	self.Pmax_Gen := Pmax_Gen_;
	self.Qmax_Turb := Qmax_Turb_;
	self.fDispo := fDispo_;
//	Self.frv_ef10 := frv_ef10_;
  self.ca_filtr := ca_filtr;
  self.cb_filtr := cb_filtr;
	Self.ValorDelAgua:= copy ( ValorDelAgua );
  self.centralesLagoArriba := centralesLagoArriba;
end;

procedure TFichaHidroConEmbalseValorizado.WriteToText( f: TArchiTexto );
begin
	inherited WriteToText(f);
	f.wr('hmin', hmin, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
	f.wr('hmax', hmax, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
	f.wr('Vmax', Vmax, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
	f.wr('Vhmed', Vhmed, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
	f.wr('hDescarga', hDescarga, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
	f.wr('caQE', caQE, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
	f.wr('cbQE', cbQE, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
	f.wr('ren', ren, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
	f.wr('Pmax_Gen', Pmax_Gen, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
	f.wr('Qmax_Turb', Qmax_Turb, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
	f.wr('fDispo', fDispo, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
//	f.wr('frv_ef10', frv_ef10, 12, 2 );
  f.wr('ca_filtr', ca_filtr, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES);
  f.wr('cb_filtr', cb_filtr, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES);
	f.wr('valordelagua', valorDelAgua, uconstantesSimSEE.CF_PRECISION, uconstantesSimSEE.CF_DECIMALES );
  f.wr('centralesLagoArriba', centralesLagoArriba);
end;

Constructor TFichaHidroConEmbalseValorizado.Create_ReadFromText( f: TArchiTexto );
begin
	inherited Create_ReadFromText(f);
  f.IniciarLecturaRetrasada;
	f.rd('hmin', hmin );
	f.rd('hmax', hmax );
	f.rd('Vmax', Vmax );
	f.rd('Vhmed', Vhmed );
	f.rd('hDescarga', hDescarga );
	f.rd('caQE', caQE );
	f.rd('cbQE', cbQE );
	f.rd('ren', ren );
	f.rd('Pmax_Gen', Pmax_Gen );
	f.rd('Qmax_Turb', Qmax_Turb );
	f.rd('fDispo', fDispo );
  f.rd('ca_filtr', ca_filtr);
  f.rd('cb_filtr', cb_filtr);
	f.rd('valordelagua', valorDelAgua );
  f.rd('centralesLagoArriba', TCosa(centralesLagoArriba));
  f.EjecutarLectura;
end;

procedure TFichaHidroConEmbalseValorizado.Free;
begin
	inherited free;
end;

procedure THidroConEmbalseValorizado_cambioFichaPD(Actor : TCosa);
begin
	(Actor as THidroConEmbalseValorizado).cambioFichaPD;
end;

procedure AlInicio;
begin
	registrarClaseDeCosa( THidroConEmbalseValorizado.ClassName, THidroConEmbalseValorizado );
	registrarClaseDeCosa( TFichaHidroConEmbalseValorizado.ClassName, TFichaHidroConEmbalseValorizado );
end;

procedure AlFinal;
begin
end;

end.


