unit uoddface_pig;

{$mode delphi}


interface

uses
  Classes, SysUtils,
  uoddface,
  Math,
  xmatdefs,
  matreal,
  uauxiliares,
  uunidades,
  uactores,
  ufechas,
  usalasdejuego,
  uDataSetGenerico,
  urosx,
  uInicioYFinal,
  uconstantessimsee;


(*
PLANIFICACION DE INVERSIONES EN GENERACION
==========================================

Se trata de determinar la expansión óptima de unidades de generación de diferentes
teconologías.
Para ello se crea una Sala SimSEE con todas las teconolgías posibles, configurando
los pagos por energía, costo variables y pagos por potencias en forma correcta para
reflejar tanto los costos fijos como variables.

A los efectos de determinar las fechas de ingreso se utilizará el mes-año
mínima. El problema consiste en determinar el cronograma óptimo de decisiones de
inversión.

Sobre cada teconología sobre la que se quiera trabajar, se definierá un registro
con:

"nombre de la máquina dentro de la sala",
"meses_construcción",
"años_vida_util",
"Costo Descisión MUSD/unidad",
"Fecha_Inicio",
"Fecha_Fin",
"MaximoUnidadesPorVez"
"MaximoUnidadesTotalActivas"


"Fecha_Inicio" y "Fecha_Fin" se utilizan como "filtro ventana para las decisiones", poner 0 (cero) si
en los límites que no apliquen y valdrán los de inicio y/o fin de simulación.

Entonces, una decisión (+n, fecha) significa agregar n unidades en la fecha+meses_construccion
y quitar n unidade sen la fecha+meses_construccion+años_vida_util

Por simplicidad, supondremos que la precisión de ingreso de las inversiones es "trimestral"
con lo cual

Para cada teconologia, se crea entonces un vector de entradas de unidades
en base al cual se actua sobre la sala.

*)


type

  { TPIG_Tecnologia }

  TPIG_Tecnologia = class
    Nombre: string; // Nombre de la máquina en SimSEE
    MesesConstruccion: integer; // meses desde la desición a la entrada en servicio
    AniosVidaUtil: integer; // años de vida útil de la unidad
    CostoDesicion_t0: NReal; // MUSD/unidad asignados al momento de toma de desicion
    // como el pago por potencia depende de la disponibilidad
    // puede ser conveniente usar este parámetro en lugar del PP.
    // En definitiva, si la inversión la pagamos igual independientemente
    // de la disponibilidad, este es el lugar de considerarla.

    // FV = parte indexada del CostoDecisión
    // tasas = tasa anual de depreciación
    // t0 = Fecha origen del Costo.
    indx_FV: NReal;
    indx_tasa: NReal;


    FechaInicio, FechaFin: TFecha;
    MaximoUnidades: integer; // máximo de unidades que se pueden decidir instalar por
    MaximoUnidadesTotalActivas: integer; // número máximo de unidades activas.
    // esta restricción, depende de la cantidad de unidades activadas
    // con anterioridad. PUede ser útil para reflejar un límite de disponibilidad
    // de combustible.

    FactorUnidadesGeneradoras_: TDAOfNInt;
    // Factor por el que se multiplica las Unidades
    // de inversión para generar las fichas de unidades de generación.
    // por ejemplo, en el caso de de los parques eólicos puede pensarse que
    // son parques de 50 MW las unidades de inversión, pero que las unidades
    // de generación para el SimSEE son molinos de 2MW y entonces el factorUnidadesGeneradoras
    // sería 25.

    class function ReadFromRec(var r: TDataRecord): TPIG_Tecnologia;
    procedure Free;
    function GetCostoDecision(dt_lapso_ini_sim: TDateTime): NReal;

  end;

  TPIG_Tecnologias = class
    lst: TList;
  private
    function getTecno(index: integer): TPIG_Tecnologia;
    procedure setTecno(index: integer; Value: TPIG_Tecnologia);

  public
    class function ReadFromDB(dbconx: TDBrosxCon;
      nid_problema: integer): TPIG_Tecnologias;
    procedure Free;
    property tecno[index: integer]: TPIG_Tecnologia read getTecno write setTecno;
      default;
  end;


  TSimCostos_archi = class
    tasa: NReal; // [p.u.] tasa de descuento en por unidad

    cad_VE: NReal; // [MUSD] valor esperado del CAD
    cad_VaR5pe: NReal; // [MUSD] VaR(5%) del CAD
    cad_CVaR5pe: NReal; // [MUSD] CVaR(5%) del CAD
    cad_VE_aux: NReal; // [MUSD] valor esperado del CAD considerando CF_aux al final
    cad_VaR5pe_aux: NReal; // [MUSD] VaR(5%) del CAD considerando CF_aux al final
    vcad: TDAOfNReal; // [MUSD] vector con los CAD de cada crónica

    // [MUSD] vector de los CAD de cada crónica, considerando CF_aux al final
    vcad_aux: TDAOfNReal;

    // [MUSD] vector con los costos directos del paso (CDP) acumulados de cada crónica.
    vcdp: TDAOfNReal;

    // [MUSD] vector con el Costo Futuro de Fin de Juego Actualizado de cad crónica.
    vcffja: TDAOfNReal;

    // [MUSD] vector con el Costo Futuro (auxiliar) de Fin de Juego Actualizado de cad crónica.
    vcffja_aux: TDAOfNReal;

    // [MUSD] vector con las utilidades directas del paso (UDP) acumuladas de cada crónica.
    vudp: TDAOfNReal;

    constructor CreateRead(archi_res: string; NCronicas: integer);
    procedure Free;

  end;


  { TPIG_Problema }

  TPIG_Problema = class(TProblema)
    CarpetaSala: string;
    NombreSala: string;

    Tecnologias: TPIG_Tecnologias;

    constructor Create(dbconx: TDBrosxCon; recProblema: TDataRecord;
      idEjecutor_: integer; tmp_rundir_: string);
    class function CreateFromDB(dbconx: TDBrosxCon; nid_problema: integer;
      idEjecutor_: integer; tmp_rundir_: string): TPIG_Problema;
    function evaluar_(Individuo: TIndividuo; SemillaAleatoria: integer): boolean;
      override;
    procedure Free; override;

    function EtapaTecnoToParametro(kEtapa, kTecno: integer): integer;
    function RunOptSim(sala: string; NCronicas, semillaAleatoria_: integer;
      Escenario: string): TSimCostos_archi;

    // Crea la sala y retorna el camino completo a la misma.
    // En la variable MontoInversiones retorna el valor a sumar al CAD
    // para considerar el monto de inversiones al Inicio del a corrida.
    // En la variable EsFactible indica que la sala resultante está dentro de las factibles.
    // En Escenario retorna el nombre del escenario activo.
    function CrearSalaSimSEE(var Escenario: string; individuo: TIndividuo;
      out MontoInversiones: NReal; out EsFactible: boolean;
      out GradoInfactibilidad: NReal): string;

    function BajarCrearSalaSimSEE(nid: integer; out aIndividuo: TIndividuo;
      out MontoInversiones: NReal): string;

  end;




implementation



function TPIG_Problema.EtapaTecnoToParametro(kEtapa, kTecno: integer): integer;
begin
  Result := kEtapa * tecnologias.lst.Count + kTecno;
end;


constructor TPIG_Problema.Create(dbconx: TDBrosxCon; recProblema: TDataRecord;
  idEjecutor_: integer; tmp_rundir_: string);

var
  kEtapa: integer;
  kTecno: integer;
  kParametro: integer;
  aTecno: TPIG_Tecnologia;

begin
  inherited Create(dbconx, recProblema, idEjecutor_, tmp_rundir_);

  // calculamos carpeta de la sala y nombre
  //  CarpetaSala := extractFilePath( ArchiSala );
  CarpetaSala := tmp_rundir_;
  NombreSala := extractFileName(ArchiSala);
  // acomodamos el directorio de la sala
  ArchiSala := tmp_rundir_ + DirectorySeparator + NombreSala;
  if (pos('.ese', NombreSala) > 0) then
    Delete(NombreSala, pos('.ese', NombreSala), length(NombreSala) -
      pos('.ese', NombreSala) + 1);


  // cargamos las posibles tecnologías
  tecnologias := TPIG_Tecnologias.ReadFromDB(dbcon, nid_Problema);
  if tecnologias = nil then
    raise Exception.Create('No pude leer archivo del problema.');

  // Bien, ahora que tenemos las tecnologias definimos los parámetros
  setlength(DescriptoresE, n_etapas * tecnologias.lst.Count);
  setlength(DescriptoresR, 0);

  for kEtapa := 0 to n_etapas - 1 do
    for kTecno := 0 to tecnologias.lst.Count - 1 do
    begin
      kParametro := EtapaTecnoToParametro(kEtapa, kTecno);
      aTecno := tecnologias[kTecno];

      self.DefinirParametroEntero(
        kParametro,
        'e' + IntToStr(kEtapa) + '_' + aTecno.Nombre,
        0, aTecno.MaximoUnidades);
    end;

  CalcularLargosADN;
end;

class function TPIG_Problema.CreateFromDB(dbconx: TDBrosxCon;
  nid_problema: integer; idEjecutor_: integer; tmp_rundir_: string): TPIG_Problema;
var
  ds: TResultadoQuery;
  r: TDataRecord;
  ap: TPIG_Problema;
  sql: string;
begin
  sql := 'SELECT * FROM ofe_problemas WHERE nid = ' + IntToStr(
    nid_problema) + ' LIMIT 1 ';
  ds := dbconx.sql_query(sql);
  r := ds.Next;
  if r <> nil then
    ap := TPIG_Problema.Create(dbconx, r, idEjecutor_, tmp_rundir_)
  else
    ap := nil;
  ds.Free;
  Result := ap;
end;


function nextpal_sep(var r: string; sep: string): string;
var
  res: string;
  i: integer;
begin
  r := trim(r);
  i := pos(sep, r);
  if i > 0 then
  begin
    res := trim(copy(r, 1, i - 1));
    Delete(r, 1, i);
  end
  else
  begin
    res := r;
    r := '';
  end;
  Result := res;
end;


function nextpal_tab(var r: string): string;
begin
  Result := nextpal_sep(r, #9);
end;




procedure readln_vect(var f: textfile; var v: TDAOfNreal; N: integer);
var
  r: string;
  pal: string;
  k: integer;
begin
  setlength(v, N);
  system.readln(f, r);
  pal := nextpal_tab(r); // caption
  for k := 0 to high(v) do
  begin
    pal := nextpal_tab(r);
    v[k] := StrToFloat(pal);
  end;
end;




class function TPIG_Tecnologia.ReadFromRec(var r: TDataRecord): TPIG_Tecnologia;
var
  res: TPIG_Tecnologia;
  s, pal: string;
  cnt: integer;

begin
  res := TPIG_Tecnologia.Create;
  res.Nombre := r.GetByNameAsString('nombre');
  res.MesesConstruccion := r.GetByNameAsInt('meses_construccion');
  res.AniosVidaUtil := r.GetByNameAsInt('anios_vida_util');
  res.CostoDesicion_t0 := r.GetByNameAsFloat('musd_por_unidad');
  res.FechaInicio := TFecha.Create_ISOStr(r.GetByNameAsString('fecha_inicio'));
  res.FechaFin := TFecha.Create_ISOStr(r.GetByNameAsString('fecha_fin'));
  res.MaximoUnidades := r.GetByNameAsInt('max_unid_por_vez');
  res.MaximoUnidadesTotalActivas := r.GetByNameAsInt('max_unid_activas');
  s := r.GetByNameAsString('factor_unidades');
  pal := nextpal_sep(s, ',');
  setlength(res.FactorUnidadesGeneradoras_, 10);
  cnt := 0;
  while pal <> '' do
  begin
    res.FactorUnidadesGeneradoras_[cnt] := StrToInt(pal);
    Inc(cnt);
    pal := nextpal_sep(s, ',');
  end;
  setlength(res.FactorUnidadesGeneradoras_, cnt);

  res.indx_FV := r.GetByNameAsFloat('indx_FV');
  res.indx_tasa := r.GetByNameAsFloat('indx_tasa');

  Result := res;
end;



procedure TPIG_Tecnologia.Free;
begin
  fechaInicio.Free;
  fechaFin.Free;
  inherited Free;
end;

function TPIG_Tecnologia.GetCostoDecision(dt_lapso_ini_sim: TDateTime): NReal;
var
  anios: NReal;
begin
  anios := dt_lapso_ini_sim / 365.2425;
  Result := CostoDesicion_t0 * (indx_FV * power(1 / (1 + indx_tasa), anios) +
    (1 - indx_FV));
end;



class function TPIG_Tecnologias.ReadFromDB(dbconx: TDBrosxCon;
  nid_problema: integer): TPIG_Tecnologias;
var
  aTecno: TPIG_Tecnologia;
  res: TPIG_Tecnologias;
  sql: string;

  ds: TResultadoQuery;
  rec: TDataRecord;

begin
  res := TPIG_Tecnologias.Create;
  res.lst := TList.Create;

  sql := 'SELECT * FROM ofe_PIG_tecnologias WHERE nid_problema = ' +
    IntToStr(NID_Problema) + ' AND activa = 1 ORDER BY nid ';
  ds := dbconx.sql_query(sql);
  rec := ds.Next;
  while rec <> nil do
  begin
    aTecno := TPIG_Tecnologia.ReadFromRec(rec);
    res.lst.add(aTecno);
    rec := ds.Next;
  end;

  ds.Free;
  Result := res;
end;



procedure TPIG_Tecnologias.Free;
var
  k: integer;
begin
  for k := 0 to lst.Count - 1 do
    tecno[k].Free;
  lst.Free;
  inherited Free;
end;

function TPIG_Tecnologias.getTecno(index: integer): TPIG_Tecnologia;
begin
  Result := lst.items[index];
end;

procedure TPIG_Tecnologias.setTecno(index: integer; Value: TPIG_Tecnologia);
begin
  lst.items[index] := Value;
end;

constructor TSimCostos_archi.CreateRead(archi_res: string; NCronicas: integer);
var
  f: textfile;

begin
  inherited Create;
  setSeparadoresGlobales;

  Assign(f, archi_res);
  {$I-}
  reset(f);
  {$I+}
  if ioresult <> 0 then
    raise Exception.Create('No encontré archivo: ' + archi_res);

  //1.00000000000000E-001  [p.u.]  Tasa de descuento anual.
  system.readln(f, tasa);

  //1.56194772422336E+003  [MUSD]  Costo esperado
  system.readln(f, cad_VE);

  //  1.64347173169052E+003  [MUSD]  Costo con riesgo 5% de ser excedido.
  system.readln(f, cad_VaR5pe);

  //  1.72340935051501E+003  [MUSD]  Costo CVaR(5%).
  system.readln(f, cad_CVaR5pe);

  //0.00000000000000E+000  [MUSD]  Costo auxiliar esperado (cdp+CFaux)
  system.readln(f, cad_VE_aux);

  //0.00000000000000E+000  [MUSD]  Costo auxiliar con riesgo 5% de ser excedido. (cdp+CFaux)
  system.readln(f, cad_VaR5pe_aux);

  //valor presente del costo (cdp+CF) por cróncia [MUSD] (Ordenado):    1.50494740485487E+003   1.50591571785433E+003   1.50612217412834E+003   1.50641847593880E+003   1.50644460396937E+003   1.50779026776910E+003   1.50788839871856E+003   1.50952644194198E+003   1.50966734338762E+003   1.50976910440591E+003   1.51044602286648E+003   1.51122191040779E+003   1.51154159746584E+003   1.51161175077217E+003   1.51181310733118E+003   1.51202502263498E+003   1.51214642916588E+003   1.51220837723692E+003   1.51261560065321E+003   1.51269612568919E+003   1.51373320456774E+003   1.51386352379762E+003   1.51393585990640E+003   1.51419562434841E+003   1.51425999004084E+003   1.51477722021026E+003   1.51478982993410E+003   1.51487385499223E+003   1.51529766808487E+003   1.51544855253895E+003   1.51622448887085E+003   1.51626414105134E+003   1.51675740914197E+003   1.51683380880476E+003   1.51700306189219E+003   1.51707317233621E+003   1.51715727965200E+003   1.51725301403590E+003   1.51726342899484E+003   1.51730075774729E+003   1.51750029375974E+003   1.51772506228803E+003   1.51837610981549E+003   1.51851595607428E+003   1.51866265078152E+003   1.51880680126115E+003   1.51884225561404E+003   1.51886043745067E+003   1.51898415821418E+003   1.51899210643537E+003   1.51911863736072E+003   1.51939005753879E+003   1.51941608530021E+003   1.51956310162697E+003   1.51957788985319E+003   1.51977421302197E+003   1.51980475958201E+003   1.51999764994909E+003   1.52030417732564E+003   1.52034003580704E+003   1.52067758755674E+003   1.52102778462024E+003   1.52107167387551E+003   1.52108418029560E+003   1.52151032310552E+003   1.52167218316628E+003   1.52198968752016E+003   1.52204682527449E+003   1.52206094245358E+003   1.52211181228827E+003   1.52225986950052E+003   1.52229528074864E+003   1.52236983633414E+003   1.52238857717461E+003   1.52241616278486E+003   1.52251176018231E+003   1.52263672637721E+003   1.52269651207963E+003   1.52280985552620E+003   1.52288133566067E+003   1.52314668284455E+003   1.52326883047941E+003   1.52328250003073E+003   1.52334650244655E+003   1.52342276442691E+003   1.52355171177731E+003   1.52356181881076E+003   1.52358023156191E+003   1.52364322176906E+003   1.52373780256454E+003   1.52377157830520E+003   1.52379378815323E+003   1.52382075439308E+003   1.52386247770342E+003   1.52418652425245E+003   1.52432479263624E+003   1.52436738802160E+003   1.52445088481804E+003   1.52454201876757E+003   1.52457636007474E+003   1.52467982743792E+003   1.52476073294446E+003   1.52478089180163E+003   1.52501902746033E+003   1.52511204102733E+003   1.52519157693777E+003   1.52521307014550E+003   1.52524727718688E+003   1.52536671110749E+003   1.52548712754578E+003   1.52554320536261E+003   1.52558080031604E+003   1.52570155222825E+003   1.52571095127463E+003   1.52581346765967E+003   1.52589152908940E+003   1.52611380855240E+003   1.52662121029141E+003   1.52664760099862E+003   1.52679812532076E+003   1.52684046893809E+003   1.52694005378877E+003   1.52702146378751E+003   1.52707141994755E+003   1.52710598381896E+003   1.52712834687867E+003   1.52716651086635E+003   1.52722225830137E+003   1.52728327829234E+003   1.52733195473606E+003   1.52735843944131E+003   1.52739814169794E+003   1.52754742477728E+003   1.52770143514393E+003   1.52771619030062E+003   1.52790665613901E+003   1.52799021602624E+003   1.52802078540750E+003   1.52804014706212E+003   1.52806109170791E+003   1.52813074991711E+003   1.52825789378300E+003   1.52837039409530E+003   1.52844483955538E+003   1.52855122501853E+003   1.52865428240816E+003   1.52882395579558E+003   1.52884052074453E+003   1.52885353103867E+003   1.52893745748056E+003   1.52894767145058E+003   1.52900160189567E+003   1.52912597027879E+003   1.52914143252970E+003   1.52919147661005E+003   1.52921500889895E+003   1.52922962821522E+003   1.52926919240211E+003   1.52930667789052E+003   1.52948255043892E+003   1.52950844529376E+003   1.52970750774622E+003   1.52973225520280E+003   1.52980450362420E+003   1.53009972455487E+003   1.53011331968077E+003   1.53020404388173E+003   1.53020932514013E+003   1.53020974114133E+003   1.53029932012883E+003   1.53031293451780E+003   1.53037240517827E+003   1.53048337419355E+003   1.53049871986470E+003   1.53052583298603E+003   1.53055353350391E+003   1.53057077814041E+003   1.53057468002855E+003   1.53066728161001E+003   1.53074526425845E+003   1.53086729562724E+003   1.53098634562466E+003   1.53103677503564E+003   1.53104339438513E+003   1.53111095760288E+003   1.53120729932372E+003   1.53124917467101E+003   1.53127373525335E+003   1.53147054719059E+003   1.53152989383754E+003   1.53159653155219E+003   1.53161131843161E+003   1.53174099810371E+003   1.53176628449392E+003   1.53185317742369E+003   1.53202359040004E+003   1.53202485730069E+003   1.53204274592728E+003   1.53221646053579E+003   1.53225323157634E+003   1.53229921144187E+003   1.53248618290089E+003   1.53250321502536E+003   1.53263619069256E+003   1.53264149204450E+003   1.53265506476360E+003   1.53270163508948E+003   1.53280897337779E+003   1.53283737490844E+003   1.53286340044105E+003   1.53288097859313E+003   1.53289365840662E+003   1.53294770270451E+003   1.53300106981435E+003   1.53306276956262E+003   1.53318379322504E+003   1.53319577704975E+003   1.53324358987301E+003   1.53333466428708E+003   1.53340377128199E+003   1.53351535642714E+003   1.53360885998560E+003   1.53370850198412E+003   1.53371079253779E+003   1.53379200700663E+003   1.53379452595810E+003   1.53380252373674E+003   1.53382482304730E+003   1.53384954885258E+003   1.53401557380623E+003   1.53402706721428E+003   1.53414844815395E+003   1.53424947226203E+003   1.53428616883161E+003   1.53431687190872E+003   1.53433484462776E+003   1.53436417183832E+003   1.53441318850163E+003   1.53450387110125E+003   1.53454793986248E+003   1.53455550289194E+003   1.53456209801131E+003   1.53456742938171E+003   1.53458409672437E+003   1.53461174418348E+003   1.53462573334411E+003   1.53471562176861E+003   1.53471702412535E+003   1.53475159698161E+003   1.53477076228523E+003   1.53478131061070E+003   1.53493537102346E+003   1.53494610166310E+003   1.53495420873962E+003   1.53501612088083E+003   1.53501706032433E+003   1.53513298678251E+003   1.53516176327788E+003   1.53519457643225E+003   1.53523753554384E+003   1.53525951797853E+003   1.53527167271185E+003   1.53536319868861E+003   1.53542500601381E+003   1.53550097759911E+003   1.53554233407106E+003   1.53561066003159E+003   1.53569278797391E+003   1.53574058937427E+003   1.53575439583094E+003   1.53578095136825E+003   1.53585502914836E+003   1.53592680980879E+003   1.53592866706068E+003   1.53593205909798E+003   1.53604949221876E+003   1.53616224191422E+003   1.53616975413971E+003   1.53620372790704E+003   1.53623640420063E+003   1.53626484501448E+003   1.53628418165048E+003   1.53631815429619E+003   1.53634446130244E+003   1.53639376860598E+003   1.53644420960109E+003   1.53650373794843E+003   1.53660586624072E+003   1.53660851509839E+003   1.53661468489083E+003   1.53666167249153E+003   1.53667135548124E+003   1.53669755425695E+003   1.53675058040867E+003   1.53677789490379E+003   1.53680665405018E+003   1.53685445429946E+003   1.53699127513690E+003   1.53704485956829E+003   1.53707031280462E+003   1.53707039827480E+003   1.53717234967775E+003   1.53722951431343E+003   1.53724319085971E+003   1.53724886275022E+003   1.53725451185912E+003   1.53725653273207E+003   1.53731272134059E+003   1.53742873136545E+003   1.53743031570839E+003   1.53751905718613E+003   1.53760379553152E+003   1.53763768008917E+003   1.53764896238801E+003   1.53771310971726E+003   1.53773887465834E+003   1.53774075543241E+003   1.53778640643764E+003   1.53786894206608E+003   1.53801488904546E+003   1.53803737175567E+003   1.53805927811480E+003   1.53811525692912E+003   1.53813176228906E+003   1.53826183956979E+003   1.53827178657844E+003   1.53828643117314E+003   1.53829403880724E+003   1.53841916738309E+003   1.53843357887719E+003   1.53849015228570E+003   1.53864049125261E+003   1.53889152789261E+003   1.53889353762191E+003   1.53896400793567E+003   1.53898819512114E+003   1.53905596156576E+003   1.53909985913522E+003   1.53912114914649E+003   1.53919168229911E+003   1.53919981574137E+003   1.53920287595919E+003   1.53928662298507E+003   1.53929598050842E+003   1.53933779532906E+003   1.53941964545863E+003   1.53948237700144E+003   1.53957849541916E+003   1.53960514937985E+003   1.53963750283593E+003   1.53963954743259E+003   1.53969283196507E+003   1.53985514541786E+003   1.53994570988598E+003   1.53998154021077E+003   1.54004869570150E+003   1.54008751525250E+003   1.54009845722911E+003   1.54020799532057E+003   1.54027893245527E+003   1.54032529344173E+003   1.54033916621446E+003   1.54037988848359E+003   1.54041606225929E+003   1.54050631703569E+003   1.54055745564989E+003   1.54065347875809E+003   1.54087446332720E+003   1.54090413752458E+003   1.54093143701698E+003   1.54097820877500E+003   1.54099777555383E+003   1.54103493292182E+003   1.54143174105130E+003   1.54151935690090E+003   1.54163901082296E+003   1.54166792188025E+003   1.54170412760179E+003   1.54176253555165E+003   1.54180340382192E+003   1.54182504372867E+003   1.54185100139995E+003   1.54185522173302E+003   1.54193704158513E+003   1.54200492561612E+003   1.54212369905593E+003   1.54214754254080E+003   1.54218335101763E+003   1.54219350754941E+003   1.54222213081864E+003   1.54230272517485E+003   1.54232755598552E+003   1.54257483390885E+003   1.54260785565079E+003   1.54264440849302E+003   1.54265819704918E+003   1.54266537247600E+003   1.54266591491747E+003   1.54276020776495E+003   1.54278542802999E+003   1.54280457767739E+003   1.54289916148931E+003   1.54290167414989E+003   1.54293902216579E+003   1.54296116224625E+003   1.54310521471510E+003   1.54311169918891E+003   1.54322000786445E+003   1.54323755069375E+003   1.54329734455867E+003   1.54333699658455E+003   1.54361382287049E+003   1.54367673334572E+003   1.54369441694686E+003   1.54375141685521E+003   1.54382453877904E+003   1.54393377040618E+003   1.54407605077107E+003   1.54414834524779E+003   1.54430748340160E+003   1.54434778348942E+003   1.54434830030421E+003   1.54438774078536E+003   1.54471942282443E+003   1.54481459372666E+003   1.54484200496573E+003   1.54485513491513E+003   1.54502050329163E+003   1.54510534459402E+003   1.54512477143377E+003   1.54517291272595E+003   1.54518413400179E+003   1.54524909953935E+003   1.54527190327788E+003   1.54536564232669E+003   1.54539695155225E+003   1.54541498386208E+003   1.54556359165357E+003   1.54559424263348E+003   1.54565950628873E+003   1.54568002508013E+003   1.54569749287394E+003   1.54577224294459E+003   1.54579145015015E+003   1.54583994257087E+003   1.54585526548019E+003   1.54595110275073E+003   1.54602042737020E+003   1.54612854286592E+003   1.54617145245798E+003   1.54618999354762E+003   1.54619440936206E+003   1.54620644538578E+003   1.54621001823567E+003   1.54623125489630E+003   1.54625515375017E+003   1.54625737225688E+003   1.54627975113443E+003   1.54630639831824E+003   1.54645977129854E+003   1.54651836038328E+003   1.54658122941317E+003   1.54658999827367E+003   1.54662047001180E+003   1.54673045748896E+003   1.54677081102157E+003   1.54677897625720E+003   1.54684545094196E+003   1.54694796485205E+003   1.54695557522599E+003   1.54697424754606E+003   1.54701775537438E+003   1.54704192897368E+003   1.54713007130558E+003   1.54714302856579E+003   1.54716772030701E+003   1.54719938440709E+003   1.54721411061594E+003   1.54724714609478E+003   1.54750335816954E+003   1.54758678387174E+003   1.54759932418633E+003   1.54760380996701E+003   1.54776342841012E+003   1.54777353719824E+003   1.54783978987020E+003   1.54785857897536E+003   1.54798432701899E+003   1.54804103494239E+003   1.54804825445067E+003   1.54811868753303E+003   1.54814812750230E+003   1.54821638625163E+003   1.54832776574093E+003   1.54835891983077E+003   1.54844514535734E+003   1.54844696442524E+003   1.54845407455638E+003   1.54847974649876E+003   1.54852133478360E+003   1.54855865026465E+003   1.54856459335831E+003   1.54858436683080E+003   1.54865346697502E+003   1.54884508289648E+003   1.54887173643862E+003   1.54890951270987E+003   1.54891095657674E+003   1.54893231363915E+003   1.54893976558040E+003   1.54898878831410E+003   1.54907653937868E+003   1.54914232292585E+003   1.54916039623570E+003   1.54929218828104E+003   1.54936250107831E+003   1.54938642862469E+003   1.54943079099653E+003   1.54968375598499E+003   1.54971247000469E+003   1.54974548789786E+003   1.54982891898983E+003   1.54983481728984E+003   1.54986762577147E+003   1.55003511095647E+003   1.55004027536375E+003   1.55015502244412E+003   1.55017029780415E+003   1.55017751777522E+003   1.55022553766182E+003   1.55031774645197E+003   1.55038580323193E+003   1.55042239945572E+003   1.55049135114020E+003   1.55058318049893E+003   1.55062843104718E+003   1.55076316487337E+003   1.55099946821283E+003   1.55111140637618E+003   1.55118029921131E+003   1.55134054004441E+003   1.55136594888153E+003   1.55142270195847E+003   1.55153103805670E+003   1.55162263501092E+003   1.55188178687887E+003   1.55190357497620E+003   1.55208464252661E+003   1.55210023507097E+003   1.55213718935829E+003   1.55227183890925E+003   1.55233646174546E+003   1.55235784358334E+003   1.55251912554612E+003   1.55255473642899E+003   1.55256837280620E+003   1.55270139910111E+003   1.55274715488187E+003   1.55275067435123E+003   1.55275320963322E+003   1.55303443675833E+003   1.55322821805449E+003   1.55328783134727E+003   1.55330301757016E+003   1.55340800519902E+003   1.55342457290919E+003   1.55343941860575E+003   1.55362942787406E+003   1.55364350871115E+003   1.55369609152941E+003   1.55378099182754E+003   1.55401569953002E+003   1.55401674220305E+003   1.55412290497556E+003   1.55423641252036E+003   1.55438973469143E+003   1.55443234553053E+003   1.55455445526479E+003   1.55474188944513E+003   1.55485563786514E+003   1.55485880066425E+003   1.55489765930416E+003   1.55502437790868E+003   1.55510705850165E+003   1.55510983188443E+003   1.55529333041851E+003   1.55540512094678E+003   1.55540611095284E+003   1.55558272613679E+003   1.55558741173734E+003   1.55573128136092E+003   1.55592604064902E+003   1.55602739395672E+003   1.55605901817515E+003   1.55622950845405E+003   1.55627245166323E+003   1.55641386133221E+003   1.55648253337156E+003   1.55661619552957E+003   1.55685613999359E+003   1.55686476177351E+003   1.55695322271951E+003   1.55702748987823E+003   1.55704946958013E+003   1.55719269053483E+003   1.55723480175064E+003   1.55724940697305E+003   1.55749052978045E+003   1.55753386697561E+003   1.55767938497879E+003   1.55781388914346E+003   1.55784400081457E+003   1.55791693469274E+003   1.55792840210571E+003   1.55796306766833E+003   1.55805372237934E+003   1.55810827581704E+003   1.55834143201500E+003   1.55834780078581E+003   1.55841671073121E+003   1.55842175944629E+003   1.55855798901264E+003   1.55869525424099E+003   1.55870097820558E+003   1.55874209376356E+003   1.55886005558480E+003   1.55886110331308E+003   1.55890703657642E+003   1.55891064422868E+003   1.55897020434163E+003   1.55899300696083E+003   1.55905542489071E+003   1.55906828052496E+003   1.55914569428476E+003   1.55932644184637E+003   1.55936275876631E+003   1.55975654963679E+003   1.55990832990593E+003   1.56003912668134E+003   1.56012703981639E+003   1.56024372452729E+003   1.56035340340218E+003   1.56045253532111E+003   1.56073093724906E+003   1.56079085645344E+003   1.56084533710181E+003   1.56085491141043E+003   1.56086264060232E+003   1.56095724251517E+003   1.56110255001258E+003   1.56111781086769E+003   1.56126915876595E+003   1.56127388711210E+003   1.56127525224481E+003   1.56131634972700E+003   1.56135098591900E+003   1.56138014482078E+003   1.56138105154247E+003   1.56142864358934E+003   1.56148284719850E+003   1.56149343683419E+003   1.56152094868749E+003   1.56171167791699E+003   1.56186301369393E+003   1.56193963427658E+003   1.56196031805411E+003   1.56196816573963E+003   1.56198334723236E+003   1.56198915046628E+003   1.56204199052281E+003   1.56208090120322E+003   1.56225667191102E+003   1.56226532472913E+003   1.56242419790884E+003   1.56246668449659E+003   1.56247534614454E+003   1.56273868831087E+003   1.56281801052188E+003   1.56292159323784E+003   1.56300487334912E+003   1.56306971027252E+003   1.56317007206001E+003   1.56320071009381E+003   1.56421531450925E+003   1.56425573367723E+003   1.56428307933619E+003   1.56435891304164E+003   1.56439726397496E+003   1.56443200462489E+003   1.56452982927660E+003   1.56491474874034E+003   1.56496092820223E+003   1.56519476305410E+003   1.56531680076274E+003   1.56531973322688E+003   1.56533546329450E+003   1.56538082871153E+003   1.56574292283969E+003   1.56585841094008E+003   1.56612798922487E+003   1.56613057642297E+003   1.56631692060371E+003   1.56644200026435E+003   1.56644957141372E+003   1.56654263414481E+003   1.56667489856079E+003   1.56699835720405E+003   1.56706725280596E+003   1.56711036975716E+003   1.56756157253977E+003   1.56764463851228E+003   1.56765813761102E+003   1.56781059935288E+003   1.56797608747483E+003   1.56805364164793E+003   1.56807361611883E+003   1.56821945006263E+003   1.56836894422183E+003   1.56853568424121E+003   1.56854408932138E+003   1.56889446661538E+003   1.56891529196460E+003   1.56959441171244E+003   1.56963546015035E+003   1.56965389930998E+003   1.56966970107357E+003   1.56982379643318E+003   1.57000464205214E+003   1.57004589874161E+003   1.57008494515393E+003   1.57008665535614E+003   1.57015210724593E+003   1.57021971495621E+003   1.57031392702619E+003   1.57033713256449E+003   1.57041150117650E+003   1.57044478823107E+003   1.57093329805542E+003   1.57095417238999E+003   1.57101768021101E+003   1.57112250156575E+003   1.57139256220663E+003   1.57161082828276E+003   1.57175902914906E+003   1.57199757991872E+003   1.57213124371690E+003   1.57216877599281E+003   1.57232694077429E+003   1.57235005045009E+003   1.57240451061648E+003   1.57243494684180E+003   1.57248868367330E+003   1.57271568267050E+003   1.57274375259276E+003   1.57274598165162E+003   1.57277635500171E+003   1.57289121601996E+003   1.57325052822307E+003   1.57325213085215E+003   1.57336430459718E+003   1.57359405656024E+003   1.57373923369413E+003   1.57404205907893E+003   1.57415345020515E+003   1.57418953894341E+003   1.57421090828667E+003   1.57428664434638E+003   1.57429937448872E+003   1.57436724431971E+003   1.57487331190427E+003   1.57533094106427E+003   1.57559700132366E+003   1.57563913807456E+003   1.57578890112852E+003   1.57587402163697E+003   1.57589089741810E+003   1.57590428149491E+003   1.57629501623973E+003   1.57669288451427E+003   1.57681361867614E+003   1.57709318820888E+003   1.57720135147402E+003   1.57732690908960E+003   1.57737551825325E+003   1.57747018596454E+003   1.57752925884551E+003   1.57753571648972E+003   1.57755993240920E+003   1.57778857755911E+003   1.57781068113809E+003   1.57809508937756E+003   1.57811698437721E+003   1.57820449212203E+003   1.57854612042416E+003   1.57919237061996E+003   1.57928503832494E+003   1.57957445040114E+003   1.57964815708732E+003   1.57972531339562E+003   1.58026942076969E+003   1.58042004365139E+003   1.58059527953055E+003   1.58071387766516E+003   1.58071540030899E+003   1.58085375388304E+003   1.58106206799713E+003   1.58106707399738E+003   1.58166712877105E+003   1.58204568313989E+003   1.58210745773171E+003   1.58220446094206E+003   1.58223832946094E+003   1.58272299857193E+003   1.58272580314359E+003   1.58315769977395E+003   1.58350317167364E+003   1.58376677436490E+003   1.58387042911813E+003   1.58416281447736E+003   1.58416438405940E+003   1.58416455341469E+003   1.58428693921726E+003   1.58442488313219E+003   1.58496973844514E+003   1.58528575926610E+003   1.58534555717632E+003   1.58535346309435E+003   1.58543902363432E+003   1.58553476896979E+003   1.58567505107392E+003   1.58622514191005E+003   1.58656152688773E+003   1.58670894195650E+003   1.58702441262226E+003   1.58727943954224E+003   1.58748631942121E+003   1.58751494928434E+003   1.58868781807396E+003   1.58997179771220E+003   1.59006345475476E+003   1.59048660303337E+003   1.59094344690375E+003   1.59106593280258E+003   1.59148701390275E+003   1.59159938811969E+003   1.59163936772884E+003   1.59175679390903E+003   1.59179905016490E+003   1.59194625381338E+003   1.59248115210323E+003   1.59257507011256E+003   1.59294129107242E+003   1.59300661064105E+003   1.59325884896842E+003   1.59386133654873E+003   1.59409199795375E+003   1.59435456790409E+003   1.59506344758974E+003   1.59519669879542E+003   1.59533186222903E+003   1.59544736191442E+003   1.59578092249933E+003   1.59666626541774E+003   1.59689010243579E+003   1.59692974586504E+003   1.59723033996064E+003   1.59777634855192E+003   1.59809322434607E+003   1.59834665466522E+003   1.59859776264971E+003   1.59900498979764E+003   1.59912499850144E+003   1.59951044970493E+003   1.59959591531924E+003   1.59985687928623E+003   1.60024467514936E+003   1.60031703788145E+003   1.60057851637423E+003   1.60072822192757E+003   1.60077501396124E+003   1.60085843388462E+003   1.60087425998982E+003   1.60141249757020E+003   1.60207942215443E+003   1.60211940438338E+003   1.60212850967060E+003   1.60308975920463E+003   1.60316719069592E+003   1.60354917352190E+003   1.60378391747655E+003   1.60393713136885E+003   1.60396140380327E+003   1.60437522611385E+003   1.60447511157522E+003   1.60464035395894E+003   1.60506021720624E+003   1.60561886830242E+003   1.60565728201494E+003   1.60624446353438E+003   1.60651615436053E+003   1.60728520412502E+003   1.60742429848041E+003   1.60801939562024E+003   1.60890459446059E+003   1.60919040256736E+003   1.60995752650238E+003   1.61065027724945E+003   1.61071227720837E+003   1.61085526058183E+003   1.61145439231396E+003   1.61153808468462E+003   1.61211540907196E+003   1.61244483729800E+003   1.61272304068300E+003   1.61337280629313E+003   1.61460796874827E+003   1.61496009689442E+003   1.61519693701682E+003   1.61524245017351E+003   1.61558482239886E+003   1.61650270500171E+003   1.61735538138254E+003   1.61756008891236E+003   1.61807397352943E+003   1.61818414248068E+003   1.61847586013454E+003   1.61893039192615E+003   1.62183115211225E+003   1.62326840636549E+003   1.62405210941448E+003   1.62539778487257E+003   1.62593028427044E+003   1.62826694793913E+003   1.62871822926792E+003   1.62893194958114E+003   1.62907851778181E+003   1.62916125369627E+003   1.63029353168521E+003   1.63138703831093E+003   1.63297918464606E+003   1.63322227581484E+003   1.63510175048344E+003   1.63620438528328E+003   1.63827941215890E+003   1.63885430179961E+003   1.63928570121183E+003   1.64093006675147E+003   1.64188081431388E+003   1.64223881670626E+003   1.64345265041164E+003   1.64383427598937E+003   1.64397132522423E+003   1.64481888033893E+003   1.64753428731403E+003   1.65024320369370E+003   1.65048970509326E+003   1.65218957980016E+003   1.65321051250490E+003   1.65360454064139E+003   1.65401379920808E+003   1.65426534248198E+003   1.65534135085537E+003   1.66272534978465E+003   1.66533240979458E+003   1.66679527812860E+003   1.66709954380411E+003   1.66903503457509E+003   1.66909888720468E+003   1.67302379668348E+003   1.68386540579958E+003   1.68843830654743E+003   1.69186804418072E+003   1.70058990526939E+003   1.70070487157316E+003   1.70327840422546E+003   1.70764781714753E+003   1.70850932498106E+003   1.72201243918016E+003   1.72287242549477E+003   1.72294962917975E+003   1.72394842610872E+003   1.72678277467150E+003   1.73613690366104E+003   1.73775611830230E+003   1.74206781531116E+003   1.74215289062228E+003   1.75102944795956E+003   1.75123189977531E+003   1.75132457607372E+003   1.77553168576415E+003   1.80120397792338E+003   1.80888388859848E+003   1.81080495938299E+003   1.81216175611386E+003   1.82897169320243E+003   1.85520447370528E+003   1.87024979372432E+003   1.88733524777671E+003   1.94939406963765E+003   1.95888815083944E+003
  readln_vect(f, vcad, NCronicas);

  //valor presente del costo auxiliar (cdp+CFAux) por cróncia [MUSD] (Ordenado):    0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000
  readln_vect(f, vcad_aux, NCronicas);

  //valor presente del costo  directos por cróncia [MUSD]:    5.28947655721562E+002   5.13603492735154E+002   4.95076759598189E+002   5.16287797733595E+002   6.28240927708384E+002   5.30963888742984E+002   5.13085682405260E+002   5.13591548478824E+002   5.18046872846375E+002   5.23842272632239E+002   5.57768132528181E+002   5.53333596059570E+002   5.33225855340311E+002   5.61605811857989E+002   4.98522157231345E+002   5.29643730249857E+002   4.99080264567633E+002   5.11814013152332E+002   5.21004885299883E+002   5.53927599108382E+002   5.29985015999607E+002   5.10899551526961E+002   5.22516735096174E+002   5.88518527861666E+002   5.60555696883147E+002   5.34255737249155E+002   4.99377345931898E+002   5.16157506010666E+002   5.09165674076171E+002   4.95228693020144E+002   5.51861526640854E+002   5.03638207803293E+002   5.03358616364107E+002   4.99501463057697E+002   4.92957935167495E+002   5.23732446532007E+002   4.98574984684867E+002   5.08228595605675E+002   5.34872368825125E+002   5.53070749845581E+002   5.12557449679630E+002   5.10041820649610E+002   5.34902255905188E+002   5.09708163265692E+002   5.12937071251494E+002   5.52122240751847E+002   4.97636780809436E+002   5.32921752294110E+002   5.20727009006790E+002   4.93051721241743E+002   5.44778554739823E+002   6.12314992676985E+002   5.14113241151197E+002   4.97195071464060E+002   5.10652212181518E+002   5.10550007500002E+002   5.71446972275674E+002   5.43772662170896E+002   5.84472583519319E+002   5.07950278325280E+002   5.08831514620789E+002   5.09315896496345E+002   5.68261249249850E+002   5.10380145415053E+002   5.08687773919752E+002   4.95792683194357E+002   6.14209561775195E+002   5.29811989931713E+002   5.30022456316556E+002   4.94261955366267E+002   5.28885104242171E+002   5.40574471462867E+002   4.99683995355491E+002   5.14609361253114E+002   5.47914553865285E+002   5.22169641789274E+002   5.02979594837317E+002   5.57503884208197E+002   5.19952530982544E+002   5.06494373233785E+002   5.04417435820914E+002   5.55460021877037E+002   5.06057987796935E+002   5.07347231915297E+002   5.94550003201378E+002   5.27452433059511E+002   5.28848947316562E+002   5.10963617582359E+002   5.19365349079458E+002   5.16212857155300E+002   5.97099424055177E+002   5.41605309211006E+002   5.03038914251481E+002   5.50050001683894E+002   5.33876399838613E+002   5.73608069412181E+002   5.14915836705209E+002   5.08467151896015E+002   5.10792005516559E+002   5.01581791488312E+002   5.51974109502016E+002   5.63488940315718E+002   5.33448405977198E+002   5.90312633140953E+002   5.08408840559441E+002   4.96040942530564E+002   5.14486259213763E+002   5.17703163034665E+002   5.20343197149729E+002   5.19997856312421E+002   5.14654629332319E+002   5.64694155876953E+002   5.16307253774014E+002   5.14906448802203E+002   5.15437763597447E+002   5.13973611704126E+002   5.12003883537562E+002   5.20516785970348E+002   5.06204462535769E+002   5.84983216409679E+002   5.13162843220907E+002   5.04451175348679E+002   5.13640052838142E+002   5.07860000396861E+002   5.11508046955014E+002   5.11044215353029E+002   5.12722312898703E+002   5.19885057948120E+002   5.38293329391984E+002   5.65435627561297E+002   5.80195967025160E+002   5.04033254016801E+002   4.87047431053158E+002   4.95919291791712E+002   5.19264972873098E+002   5.02108723919639E+002   5.47560606925337E+002   5.30422426657353E+002   5.11148130011772E+002   5.40798651612661E+002   5.13981453942072E+002   5.45625316868533E+002   5.79682070305951E+002   5.20799554876967E+002   5.71415011908340E+002   5.25699937740944E+002   5.25089705065561E+002   6.22333256384913E+002   5.93210279741592E+002   5.13144926858311E+002   5.65253915508436E+002   5.16738122511748E+002   5.10053279234303E+002   5.16843212475402E+002   5.10605806795434E+002   5.31061652047949E+002   5.26173189258795E+002   5.01020451377333E+002   5.44223752729846E+002   5.11889065684711E+002   5.15691734974440E+002   5.05032825619248E+002   5.27156054343427E+002   5.78663964090043E+002   5.83953516462266E+002   5.05254637604656E+002   5.66071064203167E+002   5.09254046876567E+002   5.20993618891330E+002   5.22539068303131E+002   5.08146156255971E+002   5.08818892377897E+002   5.21901408184656E+002   5.33785002713759E+002   5.15073844703227E+002   6.36336654633745E+002   5.28984956848684E+002   5.14697976148269E+002   5.44681620704401E+002   5.24953967286507E+002   5.70543593999551E+002   5.42025035850434E+002   6.34845703028887E+002   5.15528030812791E+002   5.30162649023286E+002   5.01976170227263E+002   4.93741488298295E+002   5.14937899654434E+002   5.04083453573670E+002   5.35504956181981E+002   5.01746194193682E+002   5.23852590412448E+002   5.17715368311533E+002   5.04622438663233E+002   4.93778577797776E+002   6.59095091348602E+002   5.73251311230749E+002   5.22531402554078E+002   4.97100075635923E+002   5.11279207562699E+002   5.26022565731851E+002   5.70727575562060E+002   5.44137425384058E+002   5.31262621006063E+002   5.75418879684793E+002   5.57205097964695E+002   5.11307666993130E+002   5.25628176525517E+002   5.14945699160428E+002   5.09685011228880E+002   5.18450743181128E+002   5.06698481050787E+002   5.46634298857112E+002   5.17924525972371E+002   5.25248255462764E+002   5.05743780384416E+002   5.43848241687954E+002   4.86615146561865E+002   5.47126090316352E+002   5.62899471670494E+002   5.17669940074739E+002   5.13037930401143E+002   4.84630591419042E+002   5.39402801263905E+002   5.19760739586874E+002   5.49960674675886E+002   5.40271133534313E+002   5.04519588852058E+002   5.59144678315980E+002   5.42680322885735E+002   5.13457926955241E+002   5.25655455962191E+002   5.18703823609564E+002   5.58435783250838E+002   5.14399474327131E+002   5.38614381832079E+002   5.40385278952774E+002   5.24544667395751E+002   5.00613753948747E+002   5.18970496234709E+002   5.17335103961921E+002   5.28246426892406E+002   5.43572275916720E+002   5.13157031926934E+002   5.33936714648753E+002   5.41064949189222E+002   5.70022219651361E+002   5.91298918442800E+002   4.98801059956343E+002   5.21924530655385E+002   5.33853732024648E+002   4.95816570270130E+002   5.15712298753044E+002   5.19699814641233E+002   5.37084303771542E+002   5.69568978133142E+002   5.19382530229506E+002   5.17672133699381E+002   5.43341353575101E+002   5.99822720553249E+002   4.89062856949975E+002   5.10869828931378E+002   5.34356557693578E+002   4.99562112567762E+002   5.22209903015891E+002   5.55547047161614E+002   5.13921918652672E+002   5.42589765985815E+002   8.60329530956514E+002   5.34284193176360E+002   5.22486371692940E+002   6.60235038736341E+002   5.02941996338511E+002   5.20101496807448E+002   5.05339288104687E+002   4.99106293286684E+002   5.15191114120961E+002   5.44789539922253E+002   5.27917098059142E+002   5.36175204837442E+002   5.05366888043671E+002   5.21592670162919E+002   5.11434464877089E+002   4.92874996923930E+002   5.38771600145623E+002   5.11887700794504E+002   5.07290042999074E+002   5.11479191492800E+002   5.01203609351211E+002   5.14361022901668E+002   6.01940348134284E+002   5.01352378975728E+002   5.30753440326056E+002   5.40592306116199E+002   5.19438997883221E+002   4.99139683157253E+002   5.10986534626421E+002   5.01823212717856E+002   5.19145329342663E+002   6.45967480528244E+002   5.11893965137666E+002   5.26089556652995E+002   4.99147996685300E+002   5.06564158636712E+002   5.29856661106040E+002   5.52597815154321E+002   5.19009694947683E+002   8.39175557436866E+002   5.09849280967455E+002   5.25058935394031E+002   5.18415777915892E+002   5.25012661401603E+002   4.97098232180770E+002   5.39872978026399E+002   4.86867337221820E+002   5.51454521187059E+002   5.28933868829344E+002   5.65195304497900E+002   5.26818405128542E+002   5.46863592771503E+002   5.12176136814209E+002   5.08297671402037E+002   5.44176858652842E+002   5.10853651596710E+002   5.07633085031904E+002   5.25553697163573E+002   6.79617526531843E+002   5.02665999173257E+002   5.36679666788977E+002   5.90060370401875E+002   5.58028180025262E+002   5.13024026380136E+002   5.04458550482613E+002   5.55133949212768E+002   5.19307089320637E+002   4.99210895024916E+002   5.20003667678158E+002   5.13535095584991E+002   5.02689088216617E+002   4.99812195769869E+002   5.12338794337190E+002   5.19912198975791E+002   5.20925002978377E+002   5.09607187620526E+002   6.92917293487745E+002   4.99719259666386E+002   6.07296251981140E+002   5.13245672046749E+002   5.46828809425419E+002   5.25595417574554E+002   5.12668154275509E+002   5.08089771118329E+002   4.87270260280535E+002   6.05653473375370E+002   5.21495641626045E+002   5.26760316702168E+002   5.14125576772866E+002   5.31043351176883E+002   5.20640954344753E+002   5.25728873965220E+002   5.30071085898461E+002   5.66429175064079E+002   5.18628099671010E+002   4.86775669602287E+002   5.00161776993757E+002   6.18854662754683E+002   5.11332478846579E+002   5.53499555647487E+002   5.02426854450833E+002   5.18951617995692E+002   5.09192139429426E+002   5.12187707114528E+002   5.04490002870390E+002   5.15770643942797E+002   5.96916661578900E+002   4.92048356203203E+002   5.14070396506438E+002   4.82552848261819E+002   5.04449870803249E+002   5.19008508006477E+002   5.05995572082402E+002   5.69114028536595E+002   4.87582380603814E+002   5.27867386976999E+002   4.90622594831972E+002   5.06925271134998E+002   5.12510534588313E+002   5.04952436731909E+002   5.30596464444122E+002   5.15643008701949E+002   4.99572357112150E+002   5.51260212219302E+002   4.96724490441628E+002   5.27353720216922E+002   5.18917854371114E+002   5.21485629372859E+002   5.12842928939414E+002   5.54760341468700E+002   5.61702378733469E+002   4.98860074582835E+002   5.05430203637772E+002   4.99755706848512E+002   5.03724955791883E+002   5.10801866792876E+002   5.22919759892453E+002   5.31124657634634E+002   5.07429576012187E+002   5.20682067015397E+002   6.14617653722406E+002   6.83060390501275E+002   5.13243682953285E+002   5.03516652515007E+002   5.23052956724659E+002   5.23163381458024E+002   5.67673652963442E+002   4.98871577184196E+002   5.36507023385882E+002   5.08968020169076E+002   5.21500861514234E+002   5.32451443833698E+002   4.87417743883736E+002   5.17461215673983E+002   5.13495288050619E+002   5.33825881404686E+002   6.60941713071967E+002   5.37118121991329E+002   5.12281784011738E+002   5.14004163856084E+002   5.37609942853852E+002   5.23554264187645E+002   5.36268177677977E+002   5.30027744867295E+002   4.95416111393673E+002   5.14498185189338E+002   4.96529531249812E+002   5.45107212036530E+002   5.62609214928159E+002   5.73717876388061E+002   5.21906294963233E+002   5.17041617298114E+002   5.60393516119103E+002   5.23028203411275E+002   6.29382723288809E+002   5.10363002487785E+002   5.44283173479780E+002   5.05118683411072E+002   5.07878926163976E+002   4.96278772452187E+002   5.06424748338433E+002   5.60889883722972E+002   5.08728725406528E+002   5.30776575106537E+002   5.22431739310005E+002   5.18034647046451E+002   5.20029828205463E+002   5.33058992401256E+002   5.05018822239107E+002   5.05836543772347E+002   5.03602891348969E+002   5.09773502932499E+002   5.33619180733084E+002   5.71641917081476E+002   5.35676991462157E+002   5.14182902308922E+002   5.17485082606812E+002   5.10366007013741E+002   5.09941519751795E+002   5.26107275842333E+002   5.52769830990201E+002   5.00950113697710E+002   5.24209559071367E+002   5.13165272180654E+002   5.08226352917795E+002   5.04032091664693E+002   4.90793130961870E+002   5.06855754352237E+002   4.97075424876709E+002   5.48294088952478E+002   5.06453963805965E+002   5.22161318425390E+002   6.59008611376961E+002   5.18587700440145E+002   5.02993001726424E+002   5.80178838524161E+002   5.05924087938967E+002   5.36021795546506E+002   5.29049602842494E+002   4.95804802104813E+002   5.30254762698317E+002   4.90852111316838E+002   5.23900734738264E+002   5.13881434997927E+002   4.97300827971081E+002   5.65480555199618E+002   4.94127558392044E+002   5.32109921035871E+002   5.55799604498248E+002   5.67764517394607E+002   5.07660504655489E+002   5.10063232153245E+002   5.08443589969402E+002   5.26050947312223E+002   5.20436267698980E+002   5.03268497390137E+002   5.34137405284031E+002   5.78699840194195E+002   6.20283735984879E+002   6.61559496523235E+002   5.40379837894122E+002   5.56111959248022E+002   5.26925207894904E+002   5.50382293973037E+002   5.11859381387593E+002   5.11262462965213E+002   4.98796008340579E+002   5.83927387970408E+002   5.15050138821270E+002   5.07008650660063E+002   5.12178040498371E+002   5.16405080320177E+002   5.43269744932683E+002   5.11283251395438E+002   5.18515555572947E+002   5.18611798233230E+002   5.63272063271549E+002   5.12207968413745E+002   5.29476706271647E+002   5.11614322819057E+002   5.43434583022895E+002   4.94420961136244E+002   5.26000072834805E+002   5.81665511411117E+002   5.43941660343054E+002   5.14579275416532E+002   4.96879170586677E+002   5.27759267995392E+002   5.41773553675681E+002   5.31457408782427E+002   5.35362300504895E+002   5.40188997173926E+002   5.07072860335970E+002   5.31889392377452E+002   5.72791387897791E+002   5.46282736544759E+002   5.31941928065616E+002   5.10927328054100E+002   5.15612425669179E+002   5.25532279492827E+002   5.01253544144652E+002   5.49890034540951E+002   5.22482781083300E+002   5.70951139233880E+002   5.14412967220769E+002   5.12880049818262E+002   5.51654735377721E+002   5.34685332585658E+002   5.68252610603398E+002   5.21138625193301E+002   5.17749083524713E+002   5.11726943300234E+002   5.62479666525698E+002   5.00303239594803E+002   5.27541267398659E+002   5.27246366269165E+002   5.46310701866286E+002   5.21827288327335E+002   4.87878020449879E+002   5.45574328617758E+002   5.10089270257920E+002   5.06776931241889E+002   5.19868602432056E+002   5.15778144492308E+002   5.04969148238893E+002   5.41443952544411E+002   5.46402937010765E+002   6.27731154766472E+002   5.55831138512277E+002   5.29950459444463E+002   8.40156503763546E+002   5.23743913747886E+002   5.02857838916810E+002   5.37241186803120E+002   5.09615788467838E+002   5.30349188004701E+002   5.16961546601014E+002   5.04143685632819E+002   5.31190965425573E+002   5.21108414434525E+002   5.14547887108036E+002   5.11321973102094E+002   5.39863092728165E+002   5.15424995676208E+002   5.32598483143024E+002   4.99476919660490E+002   5.19061590198172E+002   4.91007689373915E+002   5.15749191533049E+002   4.92923690314824E+002   4.92225956111678E+002   5.03231163108062E+002   4.85704961758372E+002   4.94965679602491E+002   5.04385808696440E+002   4.94938123917104E+002   5.07582302096943E+002   4.96056059993405E+002   5.33360926368483E+002   5.71706933796246E+002   5.57661363798537E+002   5.52958965977828E+002   5.34294364973804E+002   4.99947312987292E+002   5.15557084456852E+002   5.28524005781715E+002   5.23141616443297E+002   5.08535530413766E+002   5.20464389222225E+002   5.06189825171908E+002   5.16361103294953E+002   5.11790239416637E+002   5.22119246559820E+002   5.06707064641462E+002   5.20902546490290E+002   5.23514539971636E+002   5.07136440030167E+002   5.42010721053771E+002   5.44708789020294E+002   5.09215925958079E+002   4.90626769619849E+002   5.17898842147600E+002   4.91647437711901E+002   5.10127863548996E+002   5.25662696288451E+002   5.04753236614182E+002   6.32187240390217E+002   5.03567195801076E+002   5.12730978820470E+002   5.05943957410540E+002   5.15276660064545E+002   5.47642001032413E+002   4.96046814715142E+002   5.05798587758905E+002   5.07740790932637E+002   5.65430126047773E+002   5.74394712565570E+002   5.13599209538286E+002   5.28323334802801E+002   5.11684101873434E+002   5.39668892876392E+002   5.25060614817965E+002   4.93443340267824E+002   5.26445476315663E+002   5.96865692271779E+002   6.08545318806005E+002   5.20040319695444E+002   5.54918538560010E+002   4.93819929728724E+002   5.29741469451961E+002   7.12581325465156E+002   5.20017402574492E+002   5.03224089344829E+002   7.92239907521562E+002   5.22984693190610E+002   5.41592181258525E+002   5.22912065726833E+002   7.03114541819374E+002   5.13680641632937E+002   4.95465762547515E+002   5.09064080485880E+002   5.03485834709109E+002   5.37387144861771E+002   5.36453867155524E+002   5.46596235165277E+002   5.30110007480920E+002   5.46285681112234E+002   5.64983565255370E+002   5.17068168617557E+002   5.66890091080929E+002   5.27290546558581E+002   5.08015047679072E+002   5.87782655652347E+002   5.16504328398671E+002   5.09262042272692E+002   5.32227334902976E+002   5.17598220977901E+002   5.10728709777779E+002   5.17063708279537E+002   5.14915787300201E+002   5.37103167663434E+002   5.28308601576823E+002   5.20766798497052E+002   5.01057897465132E+002   5.15186255866714E+002   5.02866574848599E+002   5.15710090077040E+002   5.12294473805798E+002   5.10320793892558E+002   5.15886057068998E+002   5.53014984439758E+002   5.10612863326430E+002   5.32341025956633E+002   5.23336378316933E+002   5.30571510148952E+002   5.15101669056395E+002   5.02838409332065E+002   5.28207094321279E+002   5.16443475426147E+002   5.22761869261083E+002   5.02694646837242E+002   4.87197256172076E+002   5.28599865319466E+002   4.93294622875753E+002   5.23003139623898E+002   5.23418181543301E+002   5.08966304522089E+002   5.59299431390819E+002   5.67475708651191E+002   5.63205245832586E+002   5.20404102544359E+002   4.97902179596765E+002   5.06358440078137E+002   5.17449484533238E+002   5.03359354442640E+002   5.10306632052034E+002   5.30044442020727E+002   5.11183754214986E+002   5.36672955021387E+002   5.06598877885504E+002   5.23367619521155E+002   5.68292374871962E+002   4.97473640398318E+002   5.19437518626270E+002   7.61676091421540E+002   5.10001288184860E+002   5.11687720346265E+002   5.10371048173675E+002   5.07828398102038E+002   5.23748740359335E+002   5.42726188088657E+002   5.09752531117122E+002   5.07161807310301E+002   5.04626513936915E+002   5.22117293066581E+002   5.29238203713947E+002   5.19477405988480E+002   5.03386544596297E+002   5.30254235625185E+002   5.41755064149266E+002   5.04500044209979E+002   5.23193562626925E+002   5.45397180454147E+002   5.39924071003264E+002   5.50748022424301E+002   5.26279625586590E+002   4.99761786227433E+002   5.15997173150269E+002   5.21620697826363E+002   5.16339221425068E+002   5.40732663066934E+002   5.05986974497583E+002   5.03949709917646E+002   4.94699809591823E+002   5.10467042958527E+002   5.09911220406330E+002   4.98347180027788E+002   5.07462729293184E+002   5.08131663704402E+002   5.03832304378724E+002   6.62852905523962E+002   5.06121852391256E+002   5.02569876772659E+002   5.61539550033891E+002   5.04760237183985E+002   5.86118763028866E+002   5.06057554747942E+002   5.11805405864582E+002   5.46126450265616E+002   5.15763974272265E+002   5.35392579046421E+002   7.43234011636395E+002   5.62040933368832E+002   4.91979167759737E+002   5.08157076082237E+002   5.33832502592103E+002   4.91093488057969E+002   5.46374512410850E+002   5.59368316056920E+002   5.81409259908004E+002   5.05355406183235E+002   4.96428307575600E+002   5.42472291999338E+002   6.24963571047970E+002   5.16781225879952E+002   5.15035251426225E+002   5.04782907393438E+002   5.26356104271332E+002   5.30848940825780E+002   5.00698965241150E+002   5.33592359424773E+002   5.26748731847129E+002   5.30390374021706E+002   5.11977920658784E+002   5.24374080257688E+002   5.17610433030701E+002   5.38430109604269E+002   5.15174828363908E+002   5.14796644269374E+002   7.13922146902756E+002   5.23368392524928E+002   6.08985029705183E+002   5.17858448539556E+002   4.98177268298877E+002   5.16363755788496E+002   5.10787377308725E+002   5.51049980046343E+002   5.55574986277813E+002   5.14073513288084E+002   5.17218628219845E+002   5.12226005829756E+002   5.28198790615678E+002   5.30177331036469E+002   4.98425517770554E+002   5.12246215502109E+002   5.61388166227351E+002   5.51815436293884E+002   5.24661527577329E+002   5.03050503612011E+002   5.41593911511659E+002   5.04280653693944E+002   5.19655540214080E+002   5.24759492972025E+002   5.50465377919058E+002   5.37121651325737E+002   5.16442586870867E+002   4.92922228926935E+002   5.13282642714539E+002   5.36715678854118E+002   5.22619449141006E+002   5.08562114691507E+002   5.45379971782832E+002   4.99745848606078E+002   5.21072051950338E+002   5.31317536994248E+002   5.10539366274328E+002   5.14641443718013E+002   5.16901103018347E+002   5.07762456218120E+002   4.98960806784591E+002   5.04410695931044E+002   5.21198012205947E+002   5.17518427559683E+002   4.92897761559356E+002   5.17174691664420E+002   5.09599905002502E+002   5.15963896280461E+002   5.30944882119804E+002   5.02489442046880E+002   5.62282216066953E+002   5.17116270363441E+002   5.00864120414268E+002   5.74151545246156E+002   5.15388723373651E+002   5.19843271389943E+002   5.06343507916748E+002   5.91967208628295E+002   5.44854244861778E+002   4.99508153284536E+002   5.07458530758325E+002   4.83422880497505E+002   5.21260177838898E+002   4.97767796977560E+002   5.09071808677570E+002   5.09832969894850E+002   5.93201769522419E+002   5.22900670749049E+002   5.03000442982408E+002   5.08024407653920E+002   5.17221022860160E+002   5.14723107255746E+002   5.21801356745574E+002   5.25889549220358E+002   4.99482911670466E+002   5.08854677758947E+002   4.97182482962565E+002   5.28638392037784E+002   5.44306202658022E+002   5.13878719883671E+002   5.16033523967241E+002   5.06805753007933E+002   5.26580113124186E+002   5.17637891232745E+002   5.08207979664478E+002   5.23453254068185E+002   5.11425995434289E+002   5.09451985014045E+002   5.03223016902662E+002   5.02218056636652E+002   5.31190405153878E+002   5.96874127232945E+002   5.40781317550302E+002   5.16646337742135E+002   4.89314589266921E+002   5.09486512676576E+002   5.69336198800214E+002   5.00061381907222E+002   5.34561137305616E+002   5.29127740030687E+002   5.26293532379176E+002   6.46974016920779E+002   5.38508511331451E+002   5.23110021141933E+002   5.01774080290720E+002   5.15680975873409E+002   5.12970856882425E+002   5.62885598673123E+002   5.14334563855327E+002   5.11355205327344E+002   5.10822141086355E+002   5.30536075259697E+002   5.29442283006006E+002   5.07179886067173E+002   5.41445130688101E+002   5.20569172213085E+002   5.22922184070383E+002   5.42602712880007E+002   5.05427439093120E+002   5.37790656738531E+002   5.23013433535267E+002   5.11686026208900E+002   4.92244680684430E+002   5.54988305717376E+002   5.62534597850300E+002   5.11529683648633E+002   5.12665708332577E+002   5.10047419352696E+002   5.48216389098735E+002   5.54886424192726E+002   5.46520075860247E+002   5.07569392728613E+002   5.33374018440777E+002   5.63304515986660E+002   4.97806720712533E+002   5.16844201798959E+002   5.39918534548032E+002   5.00240003754366E+002   4.92392386524221E+002   5.00920450213794E+002   5.49020314728459E+002   5.01856906005451E+002   5.25819489459488E+002   4.91645255166236E+002   5.34436215030859E+002   4.98019797085115E+002   5.39408744672714E+002   5.22407038674350E+002   5.40057261174029E+002   5.22617196155921E+002   5.46473342011348E+002   5.15629726878814E+002   5.15422903221550E+002   5.20256242456155E+002   5.07162761351783E+002   5.15903646165797E+002   5.11116345579268E+002   7.52676823973158E+002   5.05710306021679E+002   5.12980940337779E+002   5.14898297255821E+002   5.23144379965029E+002   5.59430999257326E+002   5.30067594091266E+002   4.92413673840056E+002   5.12356486466090E+002   4.99222356772803E+002   4.93173769721053E+002   5.16400723447853E+002   5.78283231961525E+002   5.09286440292779E+002   4.99781693721314E+002   5.22938286730118E+002   7.71875619025409E+002   5.32826022728955E+002   5.41333072562555E+002   5.25073028560376E+002   5.19994629245385E+002   5.08881542174743E+002   5.42506946380774E+002   5.38240472299044E+002   5.42554521511098E+002   5.08348246287372E+002   5.18408692162256E+002   5.13470646576470E+002   5.54465854428138E+002   5.02457745061443E+002   5.14157017209838E+002   5.18767065026751E+002   5.23680562987526E+002   5.44621773349431E+002   5.49701712029219E+002   5.12394014659403E+002
  readln_vect(f, vcdp, NCronicas);

  //valor presente del costo futuro de cada crónica [MUSD]:    1.01909337922083E+003   1.06126981916911E+003   1.02175704920657E+003   1.04867313046864E+003   1.02224877738488E+003   1.02137257300248E+003   1.04496803997408E+003   1.02154143830369E+003   1.02751671880719E+003   1.01909674953355E+003   1.03801278997114E+003   1.04752483782505E+003   1.02750508190874E+003   1.04581848662242E+003   1.04096021977009E+003   1.03648425897501E+003   1.01767714457433E+003   1.02438971475471E+003   1.01999289025395E+003   1.02634182166131E+003   1.03139603554287E+003   1.01794096921757E+003   1.03529715404729E+003   1.02644156903276E+003   1.02216730168878E+003   1.01944035428025E+003   1.01834771635614E+003   1.01842659071370E+003   1.04427374452958E+003   1.03247274212378E+003   1.02048852380924E+003   1.03674168068030E+003   1.03942681166589E+003   1.04193027799360E+003   1.03752543902606E+003   1.02779859152469E+003   1.02521880346836E+003   1.02528676082147E+003   1.04250314942813E+003   1.03690104786661E+003   1.02633608794228E+003   1.02033058452866E+003   1.01874125280596E+003   1.04531621464298E+003   1.07250195238283E+003   1.08408214453144E+003   1.02435290671072E+003   1.03741538027038E+003   1.04446775404731E+003   1.01897330139323E+003   1.06084031356260E+003   1.02066419196908E+003   1.01872413375724E+003   1.04199661083505E+003   1.02471098650709E+003   1.04346569203002E+003   1.05763154550613E+003   1.02897109042186E+003   1.06577062017439E+003   1.02453590457561E+003   1.03074698079837E+003   1.02653913265201E+003   1.02468004182257E+003   1.01862145648061E+003   1.01930244210649E+003   1.02147074580048E+003   1.02464474002442E+003   1.02723747964842E+003   1.01934004476176E+003   1.02061189962597E+003   1.02150069898976E+003   1.03797164896130E+003   1.02282776482682E+003   1.03108813162082E+003   1.04315137893729E+003   1.01968135961068E+003   1.02967546992629E+003   1.01996630175634E+003   1.07697721488249E+003   1.02083758150227E+003   1.01885139465850E+003   1.02263506750053E+003   1.01913358914084E+003   1.04399330812911E+003   1.04372940895752E+003   1.02139264983697E+003   1.01900963165880E+003   1.04139422600098E+003   1.02919330118519E+003   1.02342669027729E+003   1.01940328094654E+003   1.01772113263536E+003   1.02069888831306E+003   1.02272635331782E+003   1.04049084448110E+003   1.02670896846927E+003   1.03136391442922E+003   1.01908027288126E+003   1.03938551225866E+003   1.02899288854024E+003   1.07718714419426E+003   1.02899221178752E+003   1.02645992392874E+003   1.11027727212844E+003   1.02190409395836E+003   1.01821904751027E+003   1.02745078237136E+003   1.01954002782505E+003   1.03690620982332E+003   1.02430962708917E+003   1.04620028207811E+003   1.08957118660502E+003   1.05799212071471E+003   1.02337998237094E+003   1.03160416537623E+003   1.01787956571957E+003   1.02542643217083E+003   1.02383151433387E+003   1.02224037701961E+003   1.07035813444569E+003   1.01909038835543E+003   1.02360991635923E+003   1.02449170945092E+003   1.03547699618769E+003   1.03654020749566E+003   1.02669654007938E+003   1.02539294403041E+003   1.03846274283769E+003   1.02418201675255E+003   1.02550781934245E+003   1.02053225490241E+003   1.02649257896923E+003   1.03398035356709E+003   1.04590575193696E+003   1.04205137685390E+003   1.02496269602791E+003   1.08566166888950E+003   1.04361963242158E+003   1.03750533696325E+003   1.02119049885362E+003   1.02230272770841E+003   1.02576724533809E+003   1.08941681689873E+003   1.01985392388112E+003   1.02525125350940E+003   1.04227614973389E+003   1.02718213384369E+003   1.02985632341525E+003   1.03218750513098E+003   1.01887866354172E+003   1.01971582293670E+003   1.01797890161360E+003   1.03671753178727E+003   1.04018427740283E+003   1.02340976701079E+003   1.06844879765698E+003   1.02194549827424E+003   1.02006372891826E+003   1.05412290193537E+003   1.03011585993141E+003   1.01811078876230E+003   1.02236531607869E+003   1.03510927038570E+003   1.02221029589978E+003   1.08907028022121E+003   1.03404134290377E+003   1.05576008790908E+003   1.02546157489204E+003   1.02167229602614E+003   1.02864123090818E+003   1.02504962079378E+003   1.05923474927003E+003   1.01980271941713E+003   1.02938506934625E+003   1.02109590943648E+003   1.01767714457433E+003   1.01994735679046E+003   1.04157447551497E+003   1.02176037955995E+003   1.04285663206637E+003   1.03864680856780E+003   1.01945781134806E+003   1.02787964675577E+003   1.02467996450778E+003   1.03992229613065E+003   1.02007065504723E+003   1.01887411235492E+003   1.02237482168616E+003   1.04243490680961E+003   1.04799821549166E+003   1.02293363324424E+003   1.01925910877646E+003   1.01770963770228E+003   1.04250763264234E+003   1.04215008926291E+003   1.06768768332290E+003   1.03326484312978E+003   1.02685502607061E+003   1.01767714457433E+003   1.03357592735243E+003   1.03476829072159E+003   1.02446912323336E+003   1.02240520876075E+003   1.01767714457433E+003   1.01844245686393E+003   1.08761378237423E+003   1.02016288019746E+003   1.02213525188460E+003   1.02439209616863E+003   1.02192630720273E+003   1.03946619151162E+003   1.02489805050140E+003   1.04516475130778E+003   1.02050905290482E+003   1.02479201990098E+003   1.02344769622564E+003   1.05639643346140E+003   1.01982945740750E+003   1.05751426364258E+003   1.03917995048393E+003   1.01811101129351E+003   1.04344460297042E+003   1.02149158270929E+003   1.03308588240940E+003   1.02209448214615E+003   1.03380609968901E+003   1.03753954760378E+003   1.02659136875083E+003   1.04297472606740E+003   1.05789819348850E+003   1.01830835753868E+003   1.03966427726469E+003   1.02039603552565E+003   1.01889112583877E+003   1.03062102896450E+003   1.06039060796556E+003   1.02404672567211E+003   1.03716701052124E+003   1.02294806486201E+003   1.02200771254029E+003   1.02108406342117E+003   1.04764447052570E+003   1.02643236613542E+003   1.01782931369773E+003   1.06747578292145E+003   1.01767714457433E+003   1.04522023052215E+003   1.03196948792269E+003   1.03556311188198E+003   1.05228637763128E+003   1.04686014564051E+003   1.02699328525607E+003   1.06535477524434E+003   1.05659520159850E+003   1.01820902664697E+003   1.03667548540123E+003   1.03402547496951E+003   1.02174751175925E+003   1.01808729001424E+003   1.08404268524633E+003   1.01882554176858E+003   1.01886242627142E+003   1.02750645600035E+003   1.01924468869339E+003   1.01945801886435E+003   1.02665741378045E+003   1.01810293864802E+003   1.03951769174589E+003   1.08906453868113E+003   1.03843148949414E+003   1.04853130851807E+003   1.06263738675843E+003   1.03137487557021E+003   1.04014222771984E+003   1.03364890701645E+003   1.02565443965777E+003   1.02056328170998E+003   1.01767714457433E+003   1.01767714457433E+003   1.02150418014135E+003   1.01821334351824E+003   1.03283967536761E+003   1.05612710766268E+003   1.03425334995474E+003   1.02365259776322E+003   1.02182080118961E+003   1.01987646786728E+003   1.02772368446639E+003   1.02573644443756E+003   1.03493116537937E+003   1.03898971861718E+003   1.04887315868609E+003   1.01838888259979E+003   1.01846311877451E+003   1.02772872242379E+003   1.02085796679183E+003   1.04757145438622E+003   1.03621415903781E+003   1.02921359048810E+003   1.07698214865151E+003   1.01820575941721E+003   1.02223820908793E+003   1.01792517565091E+003   1.03103963689480E+003   1.01870793225227E+003   1.02811758515466E+003   1.01988183294492E+003   1.11971259340257E+003   1.03739786512733E+003   1.04710984059877E+003   1.01782062628474E+003   1.02213036716419E+003   1.04771636154589E+003   1.03572402329726E+003   1.04168388779671E+003   1.03078380827388E+003   1.06639799339969E+003   1.02675094931548E+003   1.01920202224166E+003   1.05173416987820E+003   1.02791137843829E+003   1.02052628439354E+003   1.01882801469628E+003   1.01837597661851E+003   1.01773362607559E+003   1.02106677284822E+003   1.07170704954188E+003   1.02097722259580E+003   1.02771759718598E+003   1.04504138008157E+003   1.06015596245542E+003   1.02266876159377E+003   1.02894522079938E+003   1.02298303516444E+003   1.01768418581626E+003   1.04756808123228E+003   1.03686109409535E+003   1.03028944319405E+003   1.01882123488890E+003   1.02347030426086E+003   1.04255886496697E+003   1.02628221038627E+003   1.06956160005500E+003   1.02534702111909E+003   1.05831460628756E+003   1.02799693063424E+003   1.02299727970407E+003   1.03114206873861E+003   1.02171527989596E+003   1.01912400524988E+003   1.01783056558919E+003   1.02056451128984E+003   1.01767714457433E+003   1.01839863603911E+003   1.03153879513229E+003   1.02145606954946E+003   1.02180123303592E+003   1.03481505976319E+003   1.03047045203143E+003   1.06684619614734E+003   1.03524571486427E+003   1.02105714435713E+003   1.02049304947548E+003   1.01964280633651E+003   1.03317288729332E+003   1.02867962455935E+003   1.02672679926822E+003   1.03813981208135E+003   1.02134472385437E+003   1.04232363424912E+003   1.03244687139353E+003   1.05314775617997E+003   1.02279327542195E+003   1.02089102854874E+003   1.01866816081996E+003   1.02707028115752E+003   1.05045943277016E+003   1.03138301164458E+003   1.02009214796432E+003   1.01925333156331E+003   1.02062563820901E+003   1.02389258210445E+003   1.02194406133817E+003   1.03009568069133E+003   1.03323988287145E+003   1.03011958843329E+003   1.02677608839676E+003   1.02887238631539E+003   1.01831304826574E+003   1.01892442067976E+003   1.02597084825046E+003   1.04049658168973E+003   1.01872406209732E+003   1.04224069149552E+003   1.04997661224426E+003   1.02742532720388E+003   1.05144015039678E+003   1.01939310873645E+003   1.04825514776892E+003   1.01998218103120E+003   1.03746895785153E+003   1.02235610543975E+003   1.04644534201227E+003   1.03778250003793E+003   1.02320878297347E+003   1.01929774182109E+003   1.03153443192348E+003   1.05464887404887E+003   1.12745016158875E+003   1.02544893447979E+003   1.01897277758251E+003   1.02206414780104E+003   1.04894462319406E+003   1.02091266931305E+003   1.07161204824839E+003   1.01838143685171E+003   1.02394551193522E+003   1.03101352004169E+003   1.02664726598806E+003   1.05482799570855E+003   1.01849797397060E+003   1.02781068760389E+003   1.03347895949544E+003   1.03857862921179E+003   1.04233669115349E+003   1.06365689196991E+003   1.02213140448989E+003   1.02960965901441E+003   1.03829433864106E+003   1.01874846098720E+003   1.02096662407266E+003   1.07352142865460E+003   1.02488806593197E+003   1.01780102625253E+003   1.02584030508433E+003   1.01910810247272E+003   1.03428088750763E+003   1.03193940562687E+003   1.02349065658901E+003   1.02954838097556E+003   1.10640176200950E+003   1.01800672951054E+003   1.07826509385872E+003   1.01767714457433E+003   1.01767714457433E+003   1.02873086544150E+003   1.04288423870939E+003   1.02122152130756E+003   1.03080476597499E+003   1.01868456667817E+003   1.04490070246754E+003   1.01893589489815E+003   1.01798432294928E+003   1.03404999548016E+003   1.02616016534215E+003   1.08587139952489E+003   1.04082112033176E+003   1.03677131187845E+003   1.01903383502825E+003   1.03027519276900E+003   1.01966865061418E+003   1.04643205644795E+003   1.01917864640299E+003   1.02197933960530E+003   1.02003397457932E+003   1.02251497157939E+003   1.04064166074713E+003   1.05661852730126E+003   1.02012138502976E+003   1.01843994384108E+003   1.02225021222718E+003   1.01887747374662E+003   1.02148115482843E+003   1.01802885078889E+003   1.02636414869013E+003   1.01772060572250E+003   1.03396135015893E+003   1.03613079417971E+003   1.04496873815251E+003   1.03594695739165E+003   1.11652307438718E+003   1.01767714457433E+003   1.05259441001091E+003   1.03135924616045E+003   1.01840070469727E+003   1.03961734252805E+003   1.05623615642361E+003   1.02897608969682E+003   1.03989734454761E+003   1.03351527670476E+003   1.02772190027265E+003   1.01882020009155E+003   1.02247338505089E+003   1.01767714457433E+003   1.05347176579429E+003   1.02263196840926E+003   1.02198897306087E+003   1.01846062451545E+003   1.02128716679510E+003   1.02293783766111E+003   1.02259980441573E+003   1.02444040382798E+003   1.02943135807249E+003   1.03150226489509E+003   1.02427930544718E+003   1.01853049976644E+003   1.07158430819585E+003   1.07457740713781E+003   1.03718009451508E+003   1.04601655042258E+003   1.01873429839382E+003   1.04287655499539E+003   1.01888588287085E+003   1.01824598232854E+003   1.02462675608633E+003   1.02335781615461E+003   1.01957559452284E+003   1.02753928920242E+003   1.04034108504775E+003   1.02123259976899E+003   1.02694997002352E+003   1.01938403021457E+003   1.02932423429725E+003   1.02468554632544E+003   1.02101487594571E+003   1.02158655754435E+003   1.03277996563937E+003   1.02564018904007E+003   1.01864631818033E+003   1.03348569500277E+003   1.02244689159044E+003   1.03568986997142E+003   1.01887635017883E+003   1.02061530101571E+003   1.02292558899534E+003   1.01974409017415E+003   1.01857984972650E+003   1.02603312099802E+003   1.04284219161713E+003   1.01887928335103E+003   1.03468967521568E+003   1.02001418259875E+003   1.05614056168334E+003   1.06583267252721E+003   1.02218097690995E+003   1.04095445882477E+003   1.02164410706289E+003   1.02430253779701E+003   1.01832434570854E+003   1.04017342021381E+003   1.05339124055367E+003   1.03795345522671E+003   1.01822322347179E+003   1.05203469892207E+003   1.02010429377134E+003   1.06848185811026E+003   1.07520003980824E+003   1.03099856416499E+003   1.01967964784074E+003   1.03345719070155E+003   1.08135460946367E+003   1.01855719785586E+003   1.01787371646342E+003   1.01792654645678E+003   1.02373519687533E+003   1.05176676823290E+003   1.02178932293774E+003   1.03407382846956E+003   1.03959448572707E+003   1.01911459784751E+003   1.07448596547204E+003   1.01878395351900E+003   1.01791218742178E+003   1.05264804540934E+003   1.02733629668337E+003   1.02547935773843E+003   1.05218825710797E+003   1.01803386757453E+003   1.03009328996077E+003   1.02018985665829E+003   1.03421255935799E+003   1.02403270030898E+003   1.02165794678551E+003   1.01968592295176E+003   1.01954219134741E+003   1.02516299225770E+003   1.01779782288853E+003   1.05121852633977E+003   1.05553876824810E+003   1.01879134657868E+003   1.04400733638996E+003   1.01984667703564E+003   1.02010291595809E+003   1.04710430975268E+003   1.01999437136759E+003   1.04933147684055E+003   1.05537331003270E+003   1.04013907924779E+003   1.06335677002511E+003   1.05078557909498E+003   1.02815856203925E+003   1.01773044608670E+003   1.02618496944397E+003   1.02775838816252E+003   1.02328499353030E+003   1.01767714457433E+003   1.03500801785334E+003   1.07053188291001E+003   1.01953998767549E+003   1.03239449711652E+003   1.03534109517655E+003   1.04185609083463E+003   1.02273695435039E+003   1.09974294215742E+003   1.03585139051753E+003   1.01972236336923E+003   1.01917311361371E+003   1.03152328454535E+003   1.03819335196984E+003   1.05017792632299E+003   1.02202909868797E+003   1.01878006290432E+003   1.03165218993870E+003   1.01972301072212E+003   1.02550505201433E+003   1.02764317825621E+003   1.02067203969123E+003   1.02231396787946E+003   1.01914233478606E+003   1.01927350753015E+003   1.02457705115895E+003   1.02488825733184E+003   1.02376809470808E+003   1.04781513619202E+003   1.09176118571850E+003   1.02025355859200E+003   1.04561045319453E+003   1.03091049688892E+003   1.03747401428668E+003   1.01935635617163E+003   1.01874301521896E+003   1.05432845205749E+003   1.02038995898448E+003   1.02963332154196E+003   1.02956669123770E+003   1.01958458368676E+003   1.02442987483042E+003   1.03050940567597E+003   1.03952347774357E+003   1.06028494235836E+003   1.01816841050435E+003   1.05064771189321E+003   1.02069439664058E+003   1.11346712037415E+003   1.01915949604593E+003   1.04945668755384E+003   1.01799317760245E+003   1.04121270293802E+003   1.09822363391784E+003   1.01825438400395E+003   1.04308230897341E+003   1.09509534025514E+003   1.02617570304509E+003   1.06234495011033E+003   1.02554200882955E+003   1.04791490614018E+003   1.02292787346545E+003   1.02487427325953E+003   1.03714593774979E+003   1.03225475466516E+003   1.02874343156120E+003   1.03399092107555E+003   1.10700830547612E+003   1.01963548041694E+003   1.05149066743969E+003   1.02172537670113E+003   1.04924875198615E+003   1.03689382639562E+003   1.01848169638601E+003   1.05505466259345E+003   1.15437023496993E+003   1.02151056064679E+003   1.03395796559176E+003   1.02781179177836E+003   1.02960116342919E+003   1.01982482372613E+003   1.02604150643556E+003   1.02147798130578E+003   1.01800389083822E+003   1.03264864093835E+003   1.03463832244972E+003   1.01850520416184E+003   1.01767714457433E+003   1.06477806366368E+003   1.02022196902094E+003   1.02863696321118E+003   1.04752320692202E+003   1.01790594993763E+003   1.01791831361566E+003   1.02573159797602E+003   1.02461219676288E+003   1.02649254067290E+003   1.02079443873258E+003   1.02353882219622E+003   1.02327539922033E+003   1.03065400899180E+003   1.03418495562103E+003   1.02309339621911E+003   1.02414582210085E+003   1.02059301159703E+003   1.03784970609425E+003   1.04275486934301E+003   1.02372731786506E+003   1.02379592907264E+003   1.01839213491922E+003   1.03218758251193E+003   1.01954870397107E+003   1.08076607939164E+003   1.04533882029533E+003   1.03870368664395E+003   1.03644613759925E+003   1.03099566082410E+003   1.03754478308194E+003   1.01890837684691E+003   1.01882729441789E+003   1.01902557092514E+003   1.01843687686304E+003   1.04041887748888E+003   1.03773493049142E+003   1.04415246242604E+003   1.03914104449251E+003   1.03679198982778E+003   1.03952788650184E+003   1.02455421470708E+003   1.04722292388242E+003   1.02479071510421E+003   1.02467481692332E+003   1.02000267649588E+003   1.02768531308785E+003   1.05731472168884E+003   1.02304223657142E+003   1.01982437088112E+003   1.03678974350984E+003   1.04412610088323E+003   1.02062105124063E+003   1.01768512927921E+003   1.02693845490964E+003   1.06895721305911E+003   1.06516965686360E+003   1.02306380962995E+003   1.02282226960848E+003   1.03426546794015E+003   1.02678769406542E+003   1.01967147716414E+003   1.02442473802502E+003   1.02769724379659E+003   1.02350407360741E+003   1.02986722396071E+003   1.03899265032869E+003   1.02804009271670E+003   1.04935330765251E+003   1.03654936507919E+003   1.02633961109165E+003   1.01894231063234E+003   1.02394810072085E+003   1.01835073836649E+003   1.01988912170310E+003   1.01842756512179E+003   1.03785196604920E+003   1.01958909888337E+003   1.03955382228327E+003   1.05693631010064E+003   1.01858626526257E+003   1.05576205128502E+003   1.02415218639338E+003   1.01767714457433E+003   1.02054844829518E+003   1.02055418002393E+003   1.02001353190641E+003   1.11197046206888E+003   1.10329147642575E+003   1.01956242970610E+003   1.01886438770527E+003   1.02189877876882E+003   1.04171548531982E+003   1.02169910370798E+003   1.02105172759447E+003   1.01844761937822E+003   1.01989187100364E+003   1.01886936050927E+003   1.08891474631159E+003   1.04407146352712E+003   1.06003239279618E+003   1.01867554111157E+003   1.03554238604829E+003   1.07173712007474E+003   1.07362617074944E+003   1.01767714457433E+003   1.02246665875038E+003   1.02085507811988E+003   1.01813096076190E+003   1.02363273937280E+003   1.02662538795515E+003   1.02862082186560E+003   1.03400483723753E+003   1.02746958012911E+003   1.02227366853525E+003   1.11504954629968E+003   1.01953328162496E+003   1.05811451409893E+003   1.04366250014793E+003   1.01882579359332E+003   1.01858234587460E+003   1.03154017867679E+003   1.02823505827860E+003   1.01767714457433E+003   1.02067808369352E+003   1.03717110647159E+003   1.02066765257686E+003   1.01805636313449E+003   1.02468146962778E+003   1.02023713301097E+003   1.02070148720240E+003   1.01967390176978E+003   1.02397346483463E+003   1.02150992488065E+003   1.02588695386855E+003   1.02732138045294E+003   1.02093241645156E+003   1.03970721855223E+003   1.09043744404480E+003   1.02706388092645E+003   1.02202404295903E+003   1.05568865684604E+003   1.01922420023894E+003   1.02195489282930E+003   1.02440213201357E+003   1.01825501418620E+003   1.02572405414010E+003   1.05968024542341E+003   1.03551366937245E+003   1.01862078001473E+003   1.07953772358758E+003   1.02371010598770E+003   1.02205611053894E+003   1.01811595730598E+003   1.01903566910264E+003   1.03428278308842E+003   1.04095494639565E+003   1.03855853743085E+003   1.02298788947601E+003   1.03024892128520E+003   1.04075371044129E+003   1.07456290947486E+003   1.03098406857159E+003   1.03040610379920E+003   1.01992672073798E+003   1.01831306346360E+003   1.01842606370762E+003   1.01855196488594E+003   1.05456668402176E+003   1.01894612125411E+003   1.02583675369019E+003   1.03421394773314E+003   1.02140559766484E+003   1.06579603238768E+003   1.02886224081076E+003   1.03938692018364E+003   1.05108099060375E+003   1.03627368913671E+003   1.07892508753671E+003   1.02879713338851E+003   1.03168638700605E+003   1.02140619922585E+003   1.03579458349194E+003   1.01767714457433E+003   1.01767714457433E+003   1.02002783989006E+003   1.01888575272985E+003   1.02399009340458E+003   1.05978550185356E+003   1.04046279821551E+003   1.03891885943929E+003   1.02992350085640E+003   1.02043814734090E+003   1.01767714457433E+003   1.02888148788128E+003   1.03155325990450E+003   1.03541637781071E+003   1.03767562055304E+003   1.02001107115527E+003   1.02159652395972E+003   1.02350232115781E+003   1.02706415685141E+003   1.03015316436580E+003   1.03138872728082E+003   1.02280097082368E+003   1.02723135429241E+003   1.09156417931449E+003   1.01807873803450E+003   1.03713465408540E+003   1.02289378797000E+003   1.02726406773209E+003   1.04211819351374E+003   1.03023793822161E+003   1.05412668076835E+003   1.04248308825207E+003   1.02973386157754E+003   1.09078210138152E+003   1.02046169301018E+003   1.01903752139886E+003   1.02061449688389E+003   1.02417416954445E+003   1.03070587646330E+003   1.03256176324130E+003   1.05277580590183E+003   1.03082814569028E+003   1.04341427143401E+003   1.02288849764949E+003   1.06215710511369E+003   1.02194608421161E+003   1.05815078463114E+003   1.04708896539794E+003   1.06124219998902E+003   1.01943927764280E+003   1.03982166044623E+003   1.01906548325506E+003   1.01965193894073E+003   1.03341931838512E+003   1.01897722972336E+003   1.03157322117035E+003   1.01831915603274E+003   1.02625672278901E+003   1.02411218657122E+003   1.03023151310257E+003   1.02160740733444E+003   1.02678070457832E+003   1.01783883718140E+003   1.01965286557276E+003   1.04867166469911E+003   1.06262576828378E+003   1.02118538572284E+003   1.01809116922450E+003   1.04424601886665E+003   1.02143217941191E+003   1.02659177168995E+003   1.02419159081353E+003   1.02526632961792E+003   1.02728452652425E+003   1.01852829402993E+003   1.01880076770024E+003   1.02148982561817E+003   1.01928096066218E+003   1.06368101453192E+003   1.02774798376977E+003   1.02286433206381E+003   1.04012149215495E+003   1.02384058501484E+003   1.01915158373189E+003   1.02007807437756E+003   1.03249091242572E+003   1.04131698514698E+003   1.03441410028617E+003   1.02303210257469E+003   1.05620706462532E+003   1.01784140575563E+003   1.01822635898594E+003   1.04704133702076E+003   1.03326948136718E+003   1.03969399924412E+003   1.03313311600254E+003   1.02385046721128E+003   1.04087173158840E+003   1.02742524422582E+003   1.02102185462736E+003   1.02133815121049E+003   1.03443980872147E+003   1.05220699654141E+003   1.01873426235296E+003   1.01963654717873E+003   1.04028613708845E+003   1.01927421234202E+003   1.04618187672179E+003   1.01976897640535E+003   1.02266356780379E+003   1.04981943603084E+003   1.01887319844001E+003   1.02260486480276E+003   1.01871463725485E+003   1.02339275181633E+003   1.03820750336731E+003   1.05977988164660E+003   1.01828012722348E+003   1.02681144734067E+003   1.02880414503641E+003   1.01767714457433E+003   1.03718207761479E+003   1.02391391089178E+003   1.03583305694057E+003   1.02427734082184E+003
  readln_vect(f, vcffja, NCronicas);

  //valor presente del costo  futuro AUXiliar final de cada crónica [MUSD]:    0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000
  readln_vect(f, vcffja_aux, NCronicas);

  //valor presente de la utilidad directa de cada crónica [MUSD]:    0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000   0.00000000000000E+000
  readln_vect(f, vudp, NCronicas);

  closefile(f);
  setSeparadoresLocales;
end;



procedure TSimCostos_archi.Free;
begin
  setlength(vcad, 0);
  setlength(vcad_aux, 0);
  setlength(vcdp, 0);
  setlength(vcffja, 0);
  setlength(vcffja_aux, 0);
  inherited Free;
end;




// retorna NIL si falla algo
function TPIG_Problema.RunOptSim(sala: string; NCronicas, semillaAleatoria_: integer;
  Escenario: string): TSimCostos_archi;
var
  cmd: string;
  params: array of string;
  archi_simcosto: string;

begin

  if tmp_rundir = '' then
    archi_simcosto := getDir_run + NombreSala + DirectorySeparator
  else
    archi_simcosto := getDir_run;

  limpiarCarpeta(archi_simcosto, 'simres_*');
  limpiarCarpeta(archi_simcosto, 'simcosto_*');
  limpiarCarpeta(archi_simcosto, 'estado_fin_cron_*');


  setlength(params, 4);
  params[0] := 'sala=' + sala;
  params[1] := 'ejecutor=' + IntToStr(idEjecutor);
  params[2] := 'semilla=' + IntToStr(semillaAleatoria_);
  params[3] := 'nhilos=1';

  //  writeln( getDir_bin+'cmdopt '+params[0]+' '+params[1]+' '+params[2]+' '+params[3] );


{$IFDEF LINUX}
  cmd := getDir_bin + 'cmdopt';
{$ELSE}
  cmd := getDir_bin + 'cmdopt.exe';
{$ENDIF}
  if not RunChildAndWAIT(cmd, params) then
    raise Exception.Create('no puede correr la cmdopt');


{$IFDEF LINUX}
  cmd := getDir_bin + 'cmdsim';
{$ELSE}
  cmd := getDir_bin + 'cmdsim.exe';
{$ENDIF}
  setlength(params, 5);
  params[0] := 'sala=' + sala;
  params[1] := 'ejecutor=' + IntToStr(idEjecutor);
  params[2] := 'semilla=' + IntToStr(semillaAleatoria_);
  params[3] := 'NCronicasSim=' + IntToStr(NCronicas);
  params[4] := 'nhilos=1';

  if not RunChildAndWAIT(cmd, params) then
    raise Exception.Create('no puede correr la cmdsim');
  setlength(params, 0);


  try
    if tmp_rundir = '' then
      archi_simcosto := getDir_run + NombreSala + DirectorySeparator
    else
      archi_simcosto := getDir_run;

    archi_simcosto := archi_simcosto + DirectorySeparator + 'simcosto_' +
      IntToStr(semillaAleatoria_) + 'x' + IntToStr(NCronicas) + '_' + Escenario + '.xlt';
    Result := TSimCostos_archi.CreateRead(archi_simcosto, NCronicas);
  except
    Result := nil;
    raise Exception.Create('Error al leer archivo de resultados uplanes.425: ' +
      archi_simcosto);
  end;

end;


function TPIG_Problema.CrearSalaSimSEE(var Escenario: string;
  individuo: TIndividuo; out MontoInversiones: NReal; out EsFactible: boolean;
  out GradoInfactibilidad: NReal): string;
var
  sala: TSalaDejuego;
  delta_unidades: TDAOfNInt;
  kEtapa, kTecno: integer;
  kParametro: integer;
  actor: TActor;
  tecno: TPIG_Tecnologia;
  du: integer;

  afu: TFichaUnidades;
  nUnidades_, nUnidadesEnMantenimiento: TDAofNInt;
  Incertidumbres_Alta, Incertidumbres_InicioCronica: TDAOfBoolean;

  total_unidades: integer;
  fecha: TFecha;
  kEtapa_InicioOperacion, kEtapa_FinVidaUtil: integer;
  dt_InicioEtapa: double;
  dt_InicioSim_o_GuradaSim: double;
  dt_FinOpt: double;

  qActualizadorDiario: double;

  archiSalaModificada: string;
  MaxUnidadesGeneradorasActivas: integer;
  kTipoUnidad: integer;
  MontoPorUnidadDeInversion: NReal;
  FactorDeOperacionEfectiva, FactorActualizacionAlInicio: NReal;

  {$IFDEF SAQUE_INVERSIONES_XLT}
  sal: textfile;


  procedure wd(tit, val: string);
  begin
    Write(sal, #9, tit, #9, val);
  end;

  {$ENDIF}

begin

  {$IFDEF SAQUE_INVERSIONES_XLT}
  assignfile(sal, 'oddface_pig_inversiones.xlt');
  rewrite(sal);
  {$ENDIF}


  EsFactible := True;
  GradoInfactibilidad := 0;

  uInicioYFinal.AlInicio;

  // cargamos la sala de juegos_base

  chdir(carpetaSala);

  sala := TSalaDeJuego.cargarSala(0, archiSala, '__principal__', True);
  Escenario := sala.EscenarioActivo.nombre;

  dt_InicioSim_o_GuradaSim := max(sala.globs.fechaIniSim.dt,
    sala.globs.fechaGuardaSim.dt);

  dt_FinOpt := sala.globs.fechaFinOpt.dt;

  qActualizadorDiario := power(1 / (1 + sala.globs.TasaDeActualizacion), 1 / 365.2425);

  setlength(delta_unidades, n_etapas);

  MontoInversiones := 0;

  for kTecno := 0 to tecnologias.lst.Count - 1 do
  begin
    tecno := tecnologias[kTecno];
    actor := sala.ListaActores.find(tecno.Nombre) as TActor;
    vclear(delta_unidades);

  {$IFDEF SAQUE_INVERSIONES_XLT}
    writeln(sal, 'Tecnología: ', #9, tecno.Nombre);
  {$ENDIF}

    // Armamos un vector con los Delta_Unidades.
    // para tener entradas y salidas registradas.
    for kEtapa := 0 to n_etapas - 1 do
    begin
      // Chequeamos si la etapa esta en la ventana en que
      // es posible tomar decisiones para esta tecnología.
      dt_InicioEtapa := (fecha_primer_etapa.dt + kEtapa * dias_por_etapa);
      if (dt_InicioEtapa >= dt_InicioSim_o_GuradaSim) and
        (dt_InicioEtapa >= tecno.FechaInicio.dt) and
        (dt_InicioEtapa < tecno.FechaFin.dt) then
      begin
        // si estamos en-rango de tomar decisiones antomos entrada y salida.
        kParametro := EtapaTecnoToParametro(kEtapa, kTecno);
        du := Individuo.XE.e(kParametro + 1);
        if du <> 0 then
        begin
          kEtapa_InicioOperacion :=
            kEtapa + trunc((tecno.MesesConstruccion * 365.2 / 12.0) /
            dias_por_etapa + 0.2);

          if (kEtapa_InicioOperacion < n_etapas) then
          begin
            MontoPorUnidadDeInversion :=
              tecno.GetCostoDecision(dt_InicioEtapa - dt_inicioSim_o_GuradaSim);

            FactorDeOperacionEfectiva :=
              ((1 - power(qActualizadorDiario, min(
              Tecno.AniosVidaUtil * 365.2425, dt_FinOpt - dt_InicioEtapa)))
              /
              (1 - power(qActualizadorDiario, Tecno.AniosVidaUtil * 365.2425)));

            FactorActualizacionAlInicio :=
              power(qActualizadorDiario, dt_InicioEtapa - dt_InicioSim_o_GuradaSim);

            MontoInversiones :=
              MontoInversiones + MontoPorUnidadDeInversion
              // rch@201806011651 Cornalino objeta que se paguen todas las inversiones si se termina el mundo
              // y propone dejar para los sobrevientes el pago de  las mismas.

              * factorDeOperacionEfectiva

              //          MontoInversiones:= MontoInversiones + tecno.CostoDesicion *  power( qActualizadorDiario, dt_InicioEtapa - dt_InicioSim );
              // rch@20121209, Eliana detecta error en calculo de inversiones. Faltaba multiplicar por cantidad de unidades de inversión.
              * du * FactorActualizacionAlInicio;

            {$IFDEF SAQUE_INVERSIONES_XLT}
            wd('etapa: ', IntToStr(kEtapa));
            wd('Fecha:', DateTimeToStr(dt_InicioEtapa));
            wd('du: ', IntToStr(du));
            wd('MI: ', FloatToStr(MontoPorUnidadDeInversion));
            wd('foe: ', FloatToStr(FactorDeOperacionEfectiva));
            wd('faai: ', FloatTOStr(FactorActualizacionAlInicio));
            wd('MontoToalInv: ', FloatToStr(MontoInversiones));
            writeln(sal);
            {$ENDIF}

            delta_unidades[kEtapa_InicioOperacion] :=
              delta_unidades[kEtapa_InicioOperacion] + du;

            kEtapa_FinVidaUtil :=
              kEtapa_InicioOperacion + trunc(
              (tecno.AniosVidaUtil * 365.2) / dias_por_etapa + 0.2);

            if (kEtapa_FinVidaUtil < n_etapas) then
              delta_unidades[kEtapa_FinVidaUtil] :=
                delta_unidades[kEtapa_FinVidaUtil] - du;
          end;
        end;
      end;
    end;




    setlength(nUnidades_, length(Tecno.FactorUnidadesGeneradoras_));
    setlength(nUnidadesEnMantenimiento, length(Tecno.FactorUnidadesGeneradoras_));
    setlength(incertidumbres_Alta, length(Tecno.FactorUnidadesGeneradoras_));
    setlength(incertidumbres_InicioCronica, length(Tecno.FactorUnidadesGeneradoras_));

    total_unidades := 0;

    MaxUnidadesGeneradorasActivas := tecno.MaximoUnidadesTotalActivas;

    for kEtapa := 0 to n_etapas - 1 do
    begin
      if delta_unidades[kEtapa] <> 0 then
      begin

        fecha := TFecha.Create_Clone(fecha_primer_etapa);
        fecha.addDias(kEtapa * dias_por_etapa);

        total_unidades := total_unidades + delta_unidades[kEtapa];

        if total_unidades > MaxUnidadesGeneradorasActivas then
        begin
          EsFactible := False;
          GradoInfactibilidad :=
            GradoInfactibilidad + total_unidades - MaxUnidadesGeneradorasActivas;
        end;

        for kTipoUnidad := 0 to high(nUNidades_) do
        begin
          incertidumbres_Alta[kTipoUnidad] := True;
          incertidumbres_InicioCronica[kTipoUnidad] := True;
          nUnidades_[kTipoUnidad] :=
            Tecno.FactorUnidadesGeneradoras_[kTIpoUnidad] * total_unidades;
          nUnidadesEnMantenimiento[kTipoUnidad] := 0;
        end;

        afu := TFichaUnidades.Create(0, fecha, nil, nUnidades_,
          nUnidadesEnMantenimiento, Incertidumbres_Alta, Incertidumbres_InicioCronica);
        actor.lpdUnidades.insertar(afu);
      end;
    end;
  end;


  {$IFDEF SAQUE_INVERSIONES_XLT}
  closefile(sal);
  {$ENDIF}


  if esFactible then
  begin
    archiSalaModificada := getDir_run + DirectorySeparator + self.NombreSala +
      '_oddface_.ese';
    sala.WriteToArchi(archiSalaModificada);
  end;
  sala.Free;

  setlength(nUnidades_, 0);
  setlength(nUnidadesEnMantenimiento, 0);
  setlength(Incertidumbres_InicioCronica, 0);
  setlength(Incertidumbres_Alta, 0);


  uInicioYFinal.AlFinal;

  Result := archiSalaModificada;
end;


function TPIG_Problema.BajarCrearSalaSimSEE(nid: integer;
  out aIndividuo: TIndividuo; out MontoInversiones: NReal): string;
var
  EsFactible: boolean;
  Escenario: string;
  GradoInfactibilidad: NReal;
begin
  aIndividuo := LeerIndividuo(nid);
  decodificar_adn(aIndividuo);
  Result := CrearSalaSimSEE(Escenario, aIndividuo, MontoInversiones,
    EsFactible, GradoInfactibilidad);

end;

function TPIG_Problema.evaluar_(Individuo: TIndividuo;
  SemillaAleatoria: integer): boolean;
var
  res_SimCostos: TSimCostos_archi;
  archiSalaModificada: string;
  MontoInversiones: double;
  EsFactible: boolean;
  Escenario: string;
  GradoInfactibilidad: NReal;
  k: integer;
begin
  archiSalaModificada := crearSalaSimsee(Escenario, Individuo,
    MontoInversiones, EsFactible, GradoInfactibilidad);
  if EsFactible then
  begin
    res_SimCostos := RunOptSim(archiSalaModificada, NCronicasCronicasPorVez,
      SemillaAleatoria, Escenario);

    if res_SimCostos <> nil then
    begin
      Individuo.f_histo := TVectR.Create_FromDAofR(res_SimCostos.vcad);
      Individuo.f_histo.MasReal(MontoInversiones);
      Individuo.f_VE := res_SimCostos.cad_VE + MontoInversiones;
      Individuo.f_VaR := Individuo.f_histo.pe_VaR(pe_var);
      Individuo.f_CVaR := Individuo.f_histo.pe_CVaR(pe_var);
      Individuo.f_MIN := individuo.f_histo.e(1);
      Individuo.f_MAX := individuo.f_histo.e(individuo.f_histo.n);
      Individuo.f_Objetivo := ro_VE * Individuo.f_VE + ro_VaR *
        Individuo.f_VaR + ro_CVaR * Individuo.f_CVaR;
      res_SimCostos.Free;
      Result := True;
    end
    else
      raise Exception.Create('Falló la simulación.');
  end
  else
  begin
    {$IFDEF RECHAZAR_INFACTIBLES}
    Result := False;
    {$ELSE}

    Individuo.f_VE := 10000000 + GradoInfactibilidad + MontoInversiones;

    Individuo.f_histo := TVectR.Create_Init(NCronicasCronicasPorVez);
    for k := 1 to individuo.f_histo.n do
      individuo.f_histo.pon_e(k, Individuo.f_VE);
    Individuo.f_VaR := Individuo.f_VE;
    Individuo.f_CVaR := Individuo.f_VE;
    Individuo.f_MIN := Individuo.f_VE;
    Individuo.f_MAX := Individuo.f_VE;
    Individuo.f_Objetivo := ro_VE * Individuo.f_VE + ro_VaR *
      Individuo.f_VaR + ro_CVaR * Individuo.f_CVaR;
    Result := True;
    {$ENDIF}

  end;
end;


procedure TPIG_Problema.Free;
begin
  Tecnologias.Free;
  inherited Free;
end;

end.
